# Code for simple energy system models
import numpy as np
import numbers
import pickle

import sim_components.generic.items as B
import sim_components.thermodynamics.Heat_Transfer as HT

J_MWh = 2.7777778e-10
Cp_W =4187

class PeakProducer(B.Node):
    """Class implementing a peak producer in a general way to be used with a consumer.
    The idea is that the peak producer is set to produce the energy which the energy system cannot.
    """
    def construct_Node(self):
        """
                Args:

                """

        #This model has the peak function as calculation function.
        self.calc_funcs = [self.peak]

        #E a state variable for integrating the power the peak producer produces.
        self.add_var('E', vartype='state')
        #P a variable to set the power produced by the peak producer
        self.add_var('P'),

        #J_MWh a constant to convert J to MWh when multiplied onto a variable in W
        self.add_var('J_MWh', val=J_MWh)

    def peak(self):
        #The derivative of E is the power P and converted to MWh
        E_dot = P * J_MWh


class Consumer(B.Subsystem):
    """Class implementing a consumer in a general way. This class can be extended to create more specific consumer types.
        """
    def construct_Subsystem(self, P=0, T_set=20, VV=False, **c):
        """
                Args:
                    P (float): Static power in W
                    T_set (float): Setpoint temperature for the consumer
                    VV (bool): Flag to set VV production and thereby VVC return
        """

        self.add_var('P', val=P, unit='W', desc='Power')
        self.add_var('P_Process', val=0, unit='W', desc='Process power available to use, but not mandatory to use.')
        self.add_var('P_demand', val=P, unit='W', desc='Power demand')
        # self.add_var('P_Peak', val=0)
        if VV:
            self.add_var('P_VVC', val=0, unit='W', desc='Power for heat loss in hot water circulation')
            self.add_var('E_VVC', vartype='state', unit='MWh', desc='Energy consumed by consumer.')
            #If VV production the diff_vvc function needs to be simulated.
            self.calc_funcs.append(self.diff_vvc)

        self.add_var('T_set', val=T_set, unit='C', desc='Setpoint temperature of consumer')
        self.add_var('T', val=T_set, unit='C', desc='Actual temperature of consumer')
        self.add_var('T_Return', val=0, unit='C', desc='Return temperature of consumer')
        self.add_var('err', val=0, unit='C', desc='Error between setpoint and actual temperature: T_set-T')

        self.add_var('E', vartype='state', unit='MWh', desc='Energy consumed')
        self.add_var('E_demand', vartype='state', unit='MWh', desc='Energy demanded')
        # self.add_var('W_MWh', val = 1.0 / 1e6 / 3600)
        self.add_var('J_MWh', val=J_MWh)



        self.add_var('P_Process_Available', val=0)
        self.add_var('P_Process_Used', val=0)

        self.add_var('P_Preheat', val=0)

        #Process flow, Temperature in and out - flow will be calculated based on power and temeperatures
        self.add_var('Process_Flow', val=18)
        self.add_var('Process_T_in', val=38)
        self.add_var('Process_T_out', val=26)

        self.add_var('E_Process_Used', vartype='state')
        self.add_var('T_Process', val=20)

        self.add_var('Cp_W', val=Cp_W)
        self.add_var('Enabled', val=0)

        #Adding a peak producer
        self.add('Peak', PeakProducer)

        #Adding the diff function that implents the model equations
        self.calc_funcs.append(self.diff)

    def diff_vvc(self):
        #The VVC power (P_VVC set from elsewhere or static) is integrated and converted to MWh
        E_VVC_dot = -P_VVC * J_MWh

    def diff(self):

        P_demand = P + P_Process
        E_dot = -P * J_MWh
        E_demand_dot = -P * J_MWh
        E_Process_Used_dot = -P_Process_Used * J_MWh
        P_Elec = -Peak.P

        err = T_set - T

        Process_Flow = abs(P_Process_Available / (Process_T_in - Process_T_out) / Cp_W)


class KB_DH(Consumer):
    def construct_Subsystem(self, P=0, T_set=20, **c):
        super().construct_Subsystem(P, T_set, VV=False, **c)

        self.calc_funcs.append(self.diff_DH_KB)
        # self.add_var('dT_safe', val=10)
        ##self.add_var('Enabled', val=1)

        self.add_var('P_Process_Available_DH', val=.5e6)
        self.add_var('P_Process_Used', val=0)
        self.add_var('T_Process', val=30)

    def diff_DH_KB(self):
        P_Process_Available = -P_Process_Available_DH * (Enabled <= 0)
        # P = -positive((T_set - dT_safe - T) * 1e5 * (Enabled < 1))
        # P_Process = -P_DH_Available
        # -positive((T_set - dT_safe - T) * .25e5 * (Enabled < 1))


"""
class KB_SCC_Reject_Heat(Consumer):
    def construct_Subsystem(self, EP=None, P=0, T_set=20,  **c):
        super().construct_Subsystem(P, T_set, VV=False, **c)

        self.EP = EP

        self.calc_funcs.append(self.diff_KB_SCC_Reject)
        #self.add_var('dT_safe', val=10)
        ##self.add_var('Enabled', val=1)

        self.add_var('P_Process_Available_DH', val=.5e6)
        self.add_var('P_Process_Used', val=0)
        self.add_var('T_Process', val=30)


    def diff_KB_SCC_Reject(self):
        P_Process_Available = -P_Process_Available_DH*(Enabled <= 0)

"""


class KB_Heat_Recovery(Consumer):
    def construct_Subsystem(self, P=0, T_set=20, A=72610 / 6, Fm2=0.0004, T_indoor=22, preheating=True, **c):
        super().construct_Subsystem(P, T_set, VV=False, **c)

        self.add_var('T_indoor', val=T_indoor)
        self.add_var('T_Process', val=T_indoor)
        # self.add_var('efficiency', val=0.8)  # Possibly a friendly overestimation with a coil.
        # It is only valid in design conditions. Plus it looks at the max possible theoretical heat transfer
        # on the minimum heat capacity rate side. A fit can be made in off-design conditions. For now it is good enough.
        self.add_var('dH_air', val=A * Fm2 * 1000 * 1.2 * 0.8)  # kW/oC

        # m3/s flow

        # 35579 #W/C rather for the whole apartment complex. One has to be careful what simulations are running
        # We are only looking at a block out of the 6 pieces. So
        # 5930 #W/C
        # Plus efficiency of the heat transfer shall be taken into account. We cannot have an infinite heat transfer surface
        # The other "P"s are in Watts, this should possibly given in Watts or compensated for this later on

        self.calc_funcs.append(self.diff_KB_Heat_Recovery)

        if preheating:
            self.calc_funcs.append(self.diff_KB_Preheating)

    def diff_KB_Heat_Recovery(self):
        P_Process_Available = -positive((T_indoor - T) * dH_air)

    def diff_KB_Preheating(self):
        # P =
        P_Preheat = positive((T - Input.T_ambient) * dH_air)


class VS_controlled_fan(Consumer):
    def construct_Subsystem(self, P=0, T_set=20, A=72610 / 6, Fm2=0.0004, T_indoor=22, preheating=True, **c):
        super().construct_Subsystem(P, T_set, **c)

        self.add_var('T_indoor', val=T_indoor)

        # self.add_var('efficiency', val=0.8)  # Possibly a friendly overestimation with a coil.
        # It is only valid in design conditions. Plus it looks at the max possible theoretical heat transfer
        # on the minimum heat capacity rate side. A fit can be made in off-design conditions. For now it is good enough.
        self.add_var('dH_air_F', val=A * 1.2 * 1000 * 0.8)  # kW/oC
        self.add_var('F_max', val=Fm2)
        self.add_var('P_', val=0)
        # m3/s flow

        # 35579 #W/C rather for the whole apartment complex. One has to be careful what simulations are running
        # We are only looking at a block out of the 6 pieces. So
        # 5930 #W/C
        # Plus efficiency of the heat transfer shall be taken into account. We cannot have an infinite heat transfer surface
        # The other "P"s are in Watts, this should possibly given in Watts or compensated for this later on

        self.calc_funcs.append(self.diff_VS_Controlled_Fan)

    def diff_VS_Controlled_Fan(self):
        P_loss_vent = max2(0, min2(0.75, positive(-Input.T_ambient) / 25)) * positive(T_indoor - Input.T_ambient) * dH_air_F * F_max
        P = positive(P_ - P_loss_vent)


# The solar collector is also charging the BH behaving like a cooling load

class KB_Solar_charging(Consumer):
    def construct_Subsystem(self, P=0, T_set=20, N_coll=100, P_Elec_Peak=300, **c):
        super().construct_Subsystem(P, T_set, VV=False, **c)

        self.calc_funcs.append(self.diff_KB_Solar_charging)
        self.add_var('a1', val=5.88)  # W/(m2K)
        self.add_var('a2', val=0.006)  # W/(m2K2)
        self.add_var('eta_0', val=0.8)  # Max collector efficiency [-]
        self.add_var('A', val=2)  # [m2] total surface of a single collector panel
        self.add_var('N_coll', val=N_coll)  # Amount of installed collector panels
        self.add_var('T_coll', val=40)  # C mean collector fluid temperature
        # self.add_var('dT_coll', val=10) #Maybe having a dT towards the Tbrine would be the most useful approach
        # Although the problem is that if we want to use it for DHW then the temperature will have to stay higher, lowering efficiency
        self.add_var('G_offset', val=1e-8)  # Offset to avoid division by zero in a way of not using "if"
        self.add_var('P_Elec_Peak', val=P_Elec_Peak)

        # self.Input = c['Input']

        # How to come up with a decent variable T_coll?

    def diff_KB_Solar_charging(self):
        # P_=50000
        T_coll = (Input.T_ambient + T) / 2
        P_ = A * (eta_0 - (a1 * (T_coll - Input.T_ambient) / (Input.G + G_offset)) - (
                    a2 * ((T_coll - Input.T_ambient) ** 2) / (Input.G + G_offset))) * (
                         Input.G + G_offset) * N_coll if Input.G > 0 else 0

        P = -positive(P_)

        P_Elec = max2(Input.G * 0.2 * A * N_coll, P_Elec_Peak)


class FanCoil(B.Subsystem):
    def construct_Subsystem(self, FC=None):
        self.add('FC', Simple_FC_func, **FC)

class Simple_FC_func(B.Subsystem):
    def construct_Subsystem(self, P_cooling_nominal=302, P_fan_nominal=35.46, R_HEX_nominal=3.3e-6,  **c):
        self.add_var('J_MWh', val=2.7777778e-10)
        self.add_var('E', vartype='state')
        self.add_var('P', val=0)
        #Added to make soft start on P_forced
        self.add_var('P_forced', vartype='state', val=0)
        self.add_var('P_cooling_nominal', val=P_cooling_nominal*1000)
        self.add_var('P_fan_nominal', val=P_fan_nominal*1000)
        self.add_var('R_HEX_nominal', val=R_HEX_nominal)
        self.add_var('ActiveFC', val=0)
        self.add_var('ActiveFC_heating', val=0)
        self.add_var('T_brine_min', val=-20)
        # Just for testing
        # self.add_var('T_brine_max', val=50)
        self.add_var('T_Brine_Out', val=7)
        self.add_var('T_Brine_In', val=7)
        self.add_var('Cp_W', val=4187)
        self.add_var('F', val=10)
        self.add_var('T_air', val=7)
        self.add_var('R_HEX', val=R_HEX_nominal)
        self.add_var('dT_nominal', val=3)

        self.add_var('E_fan', vartype='state')
        #self.add_var('P_el_fan')
        self.add_var('P_el_fan', vartype='state')

        # self.add_var('P_demand', val=10)

        self.calc_funcs = []
        self.calc_funcs.append(self.calc)
        self.calc_funcs.append(self.fan_power)

    def fan_power(self):
        #P_el_fan = abs(no_zero_div(P_fan_nominal,dT,0)*dT_nominal * P_forced / P_cooling_nominal) ** 3
        P_el_fan_ = abs(no_zero_div(P_fan_nominal, abs(dT), 0) * dT_nominal *abs(P_forced) / P_cooling_nominal) ** 2
        #Should be **3 above
        P_el_fan_dot = (P_el_fan_ - P_el_fan)*0.001

        E_fan_dot = P_el_fan * J_MWh

    def calc(self):
        # P_demand = P_cooling_nominal*1000
        T_air = Input.T_ambient
        R_HEX = R_HEX_nominal #* (P_cooling_nominal*1000 / P_) ** 2

        # Regulate flow to keep T_brine in just above T_air
        # F = max2(0, (T_Brine_In-T_air)*10)
        # Added some passive heat loss - maybe this can be circumvented...
        dT = T_Brine_In - T_air
        #P_passive = (1-ActiveFC)* (dT) * 50
        P_passive = dT * 10000/2
        # Rename forced - fan doesnt consume power for passive exchange
        # P_forced = max2(0, (T_Brine_In-T_air))/R_HEX * ActiveFC * F

        #soft start of forced convection

        #P_forced_ = ActiveFC * max2(0, dT)/dT_nominal
        #P_forced_ = positive(1 - positive(-abs(dT) + 2)) / R_HEX * (ActiveFC + ActiveFC_heating > 0)* min2(1, no_zero_div(P_fan_nominal, P_el_fan,1))
        P_forced_ = positive(1 - positive(-abs(dT) + 2)) / R_HEX * (ActiveFC + ActiveFC_heating > 0)
        P_forced_ = min2(P_forced_, P_cooling_nominal*abs(dT)/dT_nominal)
        P_forced_ = sign(dT)*P_forced_
        P_forced_ = max2(0, P_forced_)*ActiveFC + min2(0, P_forced_)*ActiveFC_heating

        #assertion()

        P_forced_dot = (P_forced_ - P_forced )*0.01

        P = P_forced + P_passive
        T_Brine_Out = T_Brine_In - no_zero_div(P, Cp_W * F, 0)

        # P is positive when the brine is releasing heat to the environment (connected to cond, cooling mode)
        E_dot = P * J_MWh


class BoreholeStorage(B.Subsystem):
    def construct_Subsystem(self, BHS=None, T0=0):
        self.add('BHS', Simple_BHS_G_func, **BHS)

#Method for getting the G function parameters from gcloud datastore or fitting it if needed.
def get_G_params(N1=3, N2=3, D=1, H=250.0, r_b=0.114/2, B=5, k_s=3.4):

    from decimal import Decimal


    #import method for fitting g func
    from sim_components.energy_storage.g_func_utilities import check_update_g_func_fit

    #call the method to get or fit g-func to params
    G_params = check_update_g_func_fit(
            # Borehole dimensions
            D=D,  # Borehole buried depth (m)
            H=H,  # Borehole length (m)
            r_b=r_b,  # Borehole radius (m)

            B=B,  # Borehole spacing (m)

            N_1=N1,
            N_2=N2,
        nSegments=6,
        #Do not force a new fit of g-function
        force_update=False,
        #Make it silent - no promts for saving or fitting
        silent=True
        )

    #Check for ver param to see if all is well
    #if not 'ver' in G_params:
    #    raise KeyError('Borehole configuration G-function not found!!')
    print('h: ',G_params)

    return G_params

def get_G_model_by_ref(ref=''):
    from sim_components.energy_storage.g_func_utilities import get_g_func_model_ref
    details= get_g_func_model_ref(ref)
    print(details)
    G_params = details['Configuration']
    print('h: ',G_params)
    #print(G_params)
    #if not 'ver' in G_params:
    #    raise KeyError('Borehole configuration G-function not found!!')

    return G_params

class BoreholeStorage_generate(B.Subsystem):
    def construct_Subsystem(self, **s):
        s["N1"] = int(s["N1"])
        s["N2"] = int(s["N2"])

        if s["N1"]*s["N2"] > 0:
            if not s['custom_reference']:

                sp = s.copy()
                #pop G_params unknown kw params
                sp.pop('scale')
                sp.pop('T0')
                sp.pop('R')
                sp.pop('custom_reference')
                G_params = get_G_params(**sp)
            else:

                G_params = get_G_model_by_ref(s['custom_reference'])


            s.pop('custom_reference')

            BHS= {'coerce_power': True, 'N': s['N1']*s['N2'], 'H': s['H'], 'k_s': s['k_s'], 'scale': s['scale'], "B": s["B"], "N_1": s["N1"], "N_2": s["N2"], 'D': s['D'],
                        'R': s['R'],
                                  'C': G_params['C'],
                                  'h': G_params['h'],
                                  'T0': [ s['T0'] for i in range(G_params['n'])],
                                'TG': s['T0']
                       }


            self.add('BHS', Simple_BHS_G_func, **BHS)
        else:
            BHS = {'coerce_power': True, 'N': s['N1'] * s['N2'], 'H': s['H'], 'k_s': s['k_s'], 'scale': s['scale'],
                   "B": s["B"], "N_1": s["N1"], "N_2": s["N2"], 'D': s['D'],
                   'R': s['R'],
                   'C': 0,
                   'h': 0,
                   'T0': [0],
                   'TG': s['T0']
                   }

            self.add('BHS', Simple_BHS_G_func, **BHS)



class Simple_BHS_G_func(B.Subsystem):
    def construct_Subsystem(self, N=1, H=1, TG=5.9, T0=[5.9, 5.9], C=[6.91e6, 901e6], h=[444, 40], q=0, k_s=3.4, R=0.1,
                            cV=2300000, scale=1, coerce_power=False, **c):

        #print('R: ', R)
        #print('Coerve: ', coerce_power)
        self.T0 = T0[0]

        # print(h)
        # print(C)
        P_max = 15 / R * N * H / scale
        P_min = 10 / R * N * H / scale
        self.add_var('J_MWh', val=2.7777778e-10)
        N_H = N * H
        self.add_var('E', vartype='state')
        self.add_var('P_')
        self.add_var('P', val=q * N_H)
        self.add_var('N_H', val=N_H)
        self.add_var('R', val=R)
        self.add_var('scale', val=scale)

        self.add_var('T_brine_min', val=-20)
        self.add_var('T_brine_max', val=40)
        self.add_var('T_Brine_Out', val=T0[0])
        self.add_var('T_Brine_In', val=T0[0])
        L = N_H * 2

        self.add_var('Cp_W', val=Cp_W)
        self.add_var('CpW_RxL', val=-1 / Cp_W / R * L)

        self.add_var('F', val=0)
        q_max = 30

        self.add_var('q_max', val=q_max)

        self.add_var('P_max', val=q_max * N_H)
        # self.add_var('P_min', val=P_min)

        # self.add_var('P__', vartype='state')
        self.add_var('kP', val=1 / 60 / 10)
        # print('P_max: ',P_max)
        t_s = H ** 2 / (9 * k_s / cV)

        # self.add_var('kW_GWh', val=1 / 3600.0 / 1e6)
        if N>0:
            lV = None
            lh = None
            for i in range(len(C)):
                C_ = abs(C[i]) * t_s
                # if i==0:
                # C_*=100
                h_ = abs(h[i]) * 2 * np.pi * k_s
                T0_ = T0[i]

                # Add volume
                tV = self.add('V' + str(i), HT.Thermal_Mass, C_, T0_)

                # Add link conductor

                if i > 0:
                    self.add('L' + str(i), HT.Thermal_Conductor, lh,
                             side1=lV,
                             side2=tV)

                lV = tV
                lh = h_

            self.add('Lg', HT.Thermal_Ground, lh, T_Ground=TG, side1=tV)

        self.calc_funcs = []
        if N >0:
            if coerce_power:
                self.calc_funcs.append(self.coerce_power)
                print('BHS coerce power on')
            else:
                self.calc_funcs.append(self.tbo)

            self.calc_funcs.append(self.calc)

        self.add_var('k_turn', val=0.75)
        self.add_var('k_u', val=1.5)
        self.add_var('f_tube', val=0.02)
        self.add_var('Dloop', val=0.026)
        self.add_var('Lloop', val=(20+H)*2)
        self.add_var('N', val=N)
        self.add_var('Ngt0', val=(N>0))
        self.add_var('eff_pump', val=0.9)

        self.add_var('N_r', val=2)
        self.add_var('N_u', val=1)
        self.add_var('N_t', val=5)
        self.add_var('rho', val=992)
        self.add_var('E_pump', vartype='state')
        self.add_var('m_loop', vartype='state')

        self.calc_funcs.append(self.pump_power)

    def pump_power(self):
        htot = N_r * f_tube * Lloop / Dloop + N_u * k_u + N_t * k_turn

        m_loop_ = no_zero_div(F,N,0)

        m_loop_dot = (m_loop_ - m_loop)*0.01

        x = 8 * m_loop**2 /(9.86 * (Dloop ** 4) * rho)
        DP = x * htot
        Qdot = F / rho
        P_pump = min2(abs(DP * Qdot / eff_pump), abs(P*0.1))
        E_pump_dot = P_pump * J_MWh

    def coerce_power(self):
        #
        # P__dot = (F * (T_Brine_In - T_Brine_Out)* Cp_W - P__)*kP
        # (max2(P_min, min2(P_max, P_)) - P__)*kP
        # P = P__
        # P_set = F * (T_Brine_In - V0.T)* Cp_W

        expo = F * CpW_RxL
        eexpo = exp(expo)

        T_Brine_Out = V0.T - (V0.T - T_Brine_In) * eexpo
        P_set = F * Cp_W * (T_Brine_In - T_Brine_Out)

        # P_max = min2(P_max, positive(T_brine_max - V0.T) / R * N_H / scale)
        # P_min = max2(-P_max, negative(T_brine_min - V0.T) / R * N_H / scale)
        # P_max = / R_b * N_H / scale

        # P=max2(P_min, min2(P_max, P_set))
        P = P_set

    def tbo(self):
        T_Brine_Out = V0.T + R * q

    def calc(self):

        q = P / N_H * scale
        V0.H_dot = q
        E_dot = P * J_MWh



class GSHP_System(B.Subsystem):
    """
    Class implementing a GSHP (Ground Sourced Heat Pump) system in a simpliefied way.
    Heat pump performance and power use is modelled using polynomials from compressor manufacturers, brine/water flow equations are presolved for steady-state.
    """
    def construct_Subsystem(self, Subcooling=True,
                            VV_ST_Vol=2000,
                            VS_ST_Vol=10000,
                            VV_Production_EMA=True,
                            VV_Production_EMB=True,
                            VV_Production_EMHW1=True,
                            VV_Production_EMHW2=True,
                            VS_Production_EMB=True,
                            VS_Production_EMB2=True,
                            VS_VV_Preheat=True,
                            N_EMB=1, month_start=10, month_stop=5,
                            UK_Process=True,
                            FanCoil = True,


                            **c):

        """
                        Args:
                            Subcooling: True enables the subcooling circuit - false disables it.
                            VV_ST_Vol: Volume in L for the hot water buffer tank.
                            VS_ST_Vol: Volume in L for the space heating buffer tank
                            VV_Production_EMA: True enables hot water production of EMA base load heat pump in switch mode
                            VV_Production_EMB: True enables hot water production of EMB
                            VV_Production_EMHW1: True enables hot water preheating from 1st (lowest temp) HX in EMHW
                            VV_Production_EMHW2: True enables hot water preheating from 2nd (medium temp) HX in EMHW
                            VS_Production_EMB: True enables space heating production for EMB circuit 1
                            VS_Production_EMB2: True enables space heating production for EMB circuit 2
                            VS_VV_Preheat: True enables use of VS (spaceheating) to preheat hot water
                            N_EMB: Number of EMBs - they capacity equations are just multiplied with this number
                            month_start: The month the HP should start operating - starts by default in month 1 (january), but can be set to start again in this month
                            month_stop: The month the HP will stop operating and producing - used to model a case where district heating is very cheap in summer time
                            UK_Process: If True process heat from KB Process will be prioritized for the subcooling circuit and secondarily go to KB.
                            **c: more arguments as dictionary

                """

        #Bind inputs to this model
        self.bind_keys(c, ['KB', 'VS', 'VV', 'BHS', 'FC'])

        ############EMA Configuration

        #Get number of EMAs from config dict
        N_EMA = c['N EMA']

        EMA_conf = self.item_fact.getConfig(c['EMA-type'])


        Power_Scale = max(EMA_conf['Nominal_Power'] * N_EMA, 500)

        # Wanted stable dt
        dt = 60
        # P_max = 500
        mass_factor = dt / 60 * Power_Scale / 500
        mass_SCC = 3000 * mass_factor

        if not 'Nominal_Power' in EMA_conf:
            EMA_conf['Nominal_Power'] = 500

        if not 'T_cond_max' in EMA_conf:
            EMA_conf['T_cond_max'] = 63

        T_EMA_Cond_Max = EMA_conf['T_cond_max']

        # Set the compressor for EMA
        c_comp_EMA = self.item_fact.getConfig(EMA_conf['Conf_Compressor'])

        if not 'Cp' in c_comp_EMA:
            c_comp_EMA['Cp'] = 1550

        # import the table to intepolate enthalpy values and to compute saturation pressure as a function of the saturation temperature
        if EMA_conf['Refrigerant'] == 'R134A_333_6_No_NaN':
            props = pickle.load(open('thermodynamics/Interp/Interp/R134A_333_6_file.dat', "rb"))
        elif EMA_conf['Refrigerant'] == 'R410A_333_6_No_NaN':
            props = pickle.load(open('thermodynamics/Interp/Interp/R410A_333_6_file.dat', "rb"))
        elif EMA_conf['Refrigerant'] == 'R1234ze_333_6_No_NaN_2':
            props = pickle.load(open('thermodynamics/Interp/Interp/R1234zee_333_6_file.dat', "rb"))
        elif EMA_conf['Refrigerant'] == 'R1234yf_333_6_No_NaN':
            props = pickle.load(open('thermodynamics/Interp/Interp/R1234yf_333_6_file.dat', "rb"))

        self.EMA_h_Tp = props['data']['Tp']['h']
        self.EMA_p_Tq = props['data']['Tq']['p']

        self.EMA_m_dot_poly = np.array(c_comp_EMA['polynom_m_dot'], dtype=np.float64)
        self.EMA_power_poly = np.array(c_comp_EMA['polynom_power'], dtype=np.float64)
        self.EMA_Q_poly = np.array(c_comp_EMA['polynom_Q'], dtype=np.float64)

        self.calc_funcs.append(self.diff_EMA)

        #More Variables
        self.add_var('N_comp_EMA', c,
                     val=(2 if EMA_conf['Dual_Circuit'] else 1) * EMA_conf['Compressors_Circuit'] * N_EMA)
        self.add_var('P_EMA_HS_max', c, val=1e6)
        self.add_var('dT_EMA_Evap', c, val=1)
        self.add_var('T_EMA_Cond_Max', c, val=T_EMA_Cond_Max)
        self.add_var('EMA_dT_Super', c, val=6)
        self.add_var('dT_EMA_Cond', c, val=1)
        self.add_var('R_evap', c, val=3.5727452555883995e-05)
        self.add_var('R_cond', c, val=0)
        self.add_var('dT_Evap_Cond', c, val=10)
        self.add_var('T_Evap_Max', c, val=20)
        self.add_var('T_EMA_Evap', c, val=0)
        self.add_var('T_EMA_Cond', c, val=40)
        self.add_var('T_EMA_Cond_int', vartype='state')
        self.add_var('P_EMA_m_dot', val=0)
        self.add_var('P_EMA_Heat', val=0)
        self.add_var('P_EMA_Elec', val=0)
        self.add_var('EMA_VV_start', val=3)
        self.add_var('EMA_VV_keep', val=1)

        self.add_var('EMA_Sub_Flow', val=1.5)
        self.add_var('EMA_Sub_Flow_P_div', val=15000)
        self.add_var('VV_Prod_EMA', val=1 if VV_Production_EMA else 0)
        self.add_var('EMA_Cond_SCC_min', val=20)
        self.add_var('Err_EMA_VV_start', val=1)
        self.add_var('Err_EMA_VV_keep', val=-1)
        self.add_var('Err_EMA_VS_start', val=1)
        self.add_var('Err_EMA_VS_keep', val=-1)
        self.add_var('Err_EMA_KB_start', val=0)
        self.add_var('Err_EMA_KB_keep', val=-2)
        print('Cp EMA: ', c_comp_EMA['Cp'])
        self.add_var('Cp_EMA', val=c_comp_EMA['Cp'])
        self.add_var('E_EMA_VS_delivered', vartype='state')
        self.add_var('P_EMA_HS_ask')#, vartype='state')

        self.add_var('non_transcritical', val=1 if True else 0)
        self.add_var('gas_cooler_exit_T', val=40)

        ###########EMB Configuration

        EMB_present = True
        EMB_conf = self.item_fact.getConfig(c['EMB-type'])
        c_comp_EMB = self.item_fact.getConfig(EMB_conf['Conf_Compressor'])

        self.EMB_m_dot_poly = np.array(c_comp_EMB['polynom_m_dot'], dtype=np.float64)
        self.EMB_power_poly = np.array(c_comp_EMB['polynom_power'], dtype=np.float64)
        self.EMB_Q_poly = np.array(c_comp_EMB['polynom_Q'], dtype=np.float64)

        if EMB_conf['Refrigerant'] == 'R134A_333_6_No_NaN':
            propsEMB = pickle.load(open('thermodynamics/Interp/Interp/R134A_333_6_file.dat', "rb"))
        elif EMB_conf['Refrigerant'] == 'R410A_333_6_No_NaN':
            propsEMB = pickle.load(open('thermodynamics/Interp/Interp/R410A_333_6_file.dat', "rb"))
        elif EMB_conf['Refrigerant'] == 'R1234ze_333_6_No_NaN_2':
            propsEMB = pickle.load(open('thermodynamics/Interp/Interp/R1234zee_333_6_file.dat', "rb"))
        elif EMB_conf['Refrigerant'] == 'R1234yf_333_6_No_NaN':
            propsEMB = pickle.load(open('thermodynamics/Interp/Interp/R1234yf_333_6_file.dat', "rb"))

        self.EMB_h_Tp = propsEMB['data']['Tp']['h']
        self.EMB_p_Tq = propsEMB['data']['Tq']['p']

        if not 'Cp' in c_comp_EMB:
            c_comp_EMB['Cp'] = 1420

        print('Cp EMB: ', c_comp_EMB['Cp'])
        self.add_var('N_comp_EMB', c,
                     val=(1 if EMB_conf['Dual_Circuit'] else 1) * EMB_conf['Compressors_Circuit'] * N_EMB)
        self.add_var('N_comp_EMB2', c,
                     val=(1 if EMB_conf['Dual_Circuit'] else 1) * EMB_conf['Compressors_Circuit'] * N_EMB)
        self.calc_funcs.append(self.diff_EMB)

        self.add_var('P_EMB_HS_max', c, val=25e3)
        self.add_var('P_EMB2_HS_max', c, val=25e3)
        self.add_var('EMB_dT_Super', c, val=6)
        self.add_var('dT_EMB_Cond', c, val=1)
        self.add_var('T_EMB_Evap', c, val=20)
        self.add_var('T_EMB_Cond', c, val=60)
        self.add_var('P_EMB_Heat', val=0)
        self.add_var('P_EMB2_Heat', val=0)
        self.add_var('P_EMB_Elec', val=0)
        self.add_var('P_EMB2_Elec', val=0)
        self.add_var('EMB_Present', val=1 if EMB_present else 0)
        self.add_var('EMB_energy', vartype='state', val=0)
        self.add_var('P_KB_EMB', val=0)
        self.add_var('VV_Prod_EMB', val=1 if VV_Production_EMB else 0)
        self.add_var('VS_Prod_EMB', val=1 if VS_Production_EMB else 0)
        self.add_var('VS_Prod_EMB2', val=1 if VS_Production_EMB2 else 0)

        print('Fan Coil: ',FanCoil)
        self.add_var('FanCoil_Active', val=1 if FanCoil else 0)
        self.add_var('EMB_VS_dT', val=0)
        self.add_var('T_EMB_Evap_Max', val=25)
        self.add_var('E_EMB1_VS', vartype='state')
        self.add_var('E_EMB_VV', vartype='state')
        self.add_var('E_EMB2_VS', vartype='state')
        self.add_var('Cp_EMB', val=c_comp_EMB['Cp'])

        ###############VV Production
        self.add_var('T_Cold', c, val=10)
        self.add_var('VV_dT_offset', c, val=5)
        self.add_var('P_VS_VV_Preheat', val=0)
        self.add_var('E_VS_VV_Preheat', vartype='state')
        self.add_var('VS_VV_Preheat', val=1 if VS_VV_Preheat else 0)
        self.add_var('R_VV', val=3e-5)

        ####VS Production

        self.add_var('T_VS_min', c, val=0)
        self.add_var('VS_dT_offset', c, val=2)
        self.add_var('R_VS', val=.3e-5)

        #KB
        self.add_var('R_KB', val=.3e-5)

        ###Sub cooling
        T_Sub = 25
        #self.add_var('T_Sub', c, val=T_Sub)
        #self.add_var('X_Sub', c, val=.2)
        self.add_var('P_Sub_EMHW1', val=0)
        self.add_var('P_Sub_EMHW2', val=0)
        self.add_var('VV_Prod_EMHW1', val=1 if VV_Production_EMHW1 else 0)
        self.add_var('VV_Prod_EMHW2', val=1 if VV_Production_EMHW2 else 0)
        self.add_var('m_SCC', val=mass_SCC)
        self.add_var('VS_SCC3_min', val=15)
        self.add_var('VS_SCC2_min', val=20)
        self.add_var('T_SCC_VV_Min', c, val=10)

        ###Other
        self.add_var('Flow_Power', c, val=10000)
        self.add_var('J_MWh', val=2.7777778e-10)
        self.add_var('Energy_Consumption', vartype='state')
        self.add_var('Energy_Consumption_Compressors', vartype='state')
        self.add_var('Energy_Consumption_FanCoil', vartype='state')
        self.add_var('Energy_Consumption_Pumps', vartype='state')
        self.add_var('Energy_Electrical_Loss', vartype='state')

        self.add_var('eta_thermal', val=0.93)

        self.add_var('BHS_E_Free', vartype='state')
        self.add_var('BHS_E_Heating', vartype='state')
        self.add_var('BHS_E_Active', vartype='state')

        self.add_var('X_Pumps', val=1.1)
        self.add_var('X_Regulation', val=1.1)

        self.add_var('Month', val=0)
        self.add_var('Month_Stop', val=month_stop)
        self.add_var('Month_Start', val=month_start)
        self.add_var('T_max_brine', val=30)

        self.add_var('E_KB_Process', vartype='state')

        self.add_var('E_Process_Sub', vartype='state')
        self.add_var('E_Process', vartype='state')
        self.add_var('i', vartype='state')
        self.add_var('P_Process_Sub_ask')
        self.add_var('dT_Process_Sub')
        self.add_var('Enabled', val=1)

        self.add_var('P_active_FC', vartype='state')
        self.add_var('P_heating_FC', vartype='state')


        self.add_var('Cp_W', val=Cp_W)

        #Construct KB Manifold
        T_KB = self.BHS.BHS.T0

        VV_ST_Vol = 10000

        self.VV_ST = self.add('VV_ST', HT.Thermal_Mass, VV_ST_Vol*2 * Cp_W, T0=60)
        # TODO: make SCC in three tanks

        self.SCC1 = self.add('SCC1', HT.Thermal_Mass, mass_SCC * Cp_W, T0=T_Sub)
        self.SCC2 = self.add('SCC2', HT.Thermal_Mass, mass_SCC * Cp_W, T0=T_Sub)
        self.SCC3 = self.add('SCC3', HT.Thermal_Mass, mass_SCC * Cp_W, T0=T_Sub)

        mass_KB = 5000 * mass_factor
        self.A1 = self.add('A1', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.A2 = self.add('A2', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.B1 = self.add('B1', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.B2 = self.add('B2', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.E = self.add('E', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.F = self.add('F', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.HP1 = self.add('HP1', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.HP2 = self.add('HP2', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)

        self.C = self.add('C', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.D = self.add('D', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)
        self.KB1 = self.add('KB1', HT.Thermal_Mass, mass_KB * Cp_W, T0=T_KB)

        mass_VS = 10000 * mass_factor

        self.VS_ST = self.add('VS_ST', HT.Thermal_Mass, mass_VS * Cp_W, T0=40)
        self.FC1 = self.add('FC1', HT.Thermal_Mass, mass_VS * Cp_W, T0=10)
        self.FC2 = self.add('FC2', HT.Thermal_Mass, mass_VS * Cp_W, T0=10)
        self.VS_ST_Return = self.add('VS_ST_Return', HT.Thermal_Mass, mass_VS * Cp_W, T0=40)

        self.calc_funcs.append(self.diff_T)
        self.calc_funcs.append(self.diff_KB_manifold)

        #Subcooling configuration
        if Subcooling:
            if UK_Process:
                self.calc_funcs.append(self.diff_w_sub_process)
            self.calc_funcs.append(self.diff_w_subcooling)

        else:
            if VS_VV_Preheat:
                self.calc_funcs.append(self.diff_VS_VV_preheat)

            self.calc_funcs.append(self.diff_wo_subcooling)

        self.calc_funcs.append(self.diff)



    def diff_T(self):
        i_dot = 1 * 1
        Enabled = min2((Month < Month_Stop) + (Month >= Month_Start), 1)

        KB.Enabled = Enabled

        # Definition of heating mode: VS_ST is below setpoint
        # VS_ST_Err = VS.err

        # Heating = -KB.err < -2
        # ((VS_ST_err > 0) + (VV_ST_err > 0) + (P_EMA_Sub_Cond_delivered > 0) > 0)

        # Definition of HP_cooling mode: HP2 is above setpoint temperature

        # TODO: 2 needs to be param
        HP_cooling = -KB.err > 0

        # Definition of active mode: VS_ST is above setpoint and HP cooling mode is enabled
        # TODO: 2 needs to be param
        # Active = HP_cooling * (VS.err < -5)
        # Active = (B2.T > max2(C.T, HP2.T))*(Heating <= 0)
        # Active = ((B2.T > KB1.T) + HP_cooling > 0) * (Heating <= 0) * (KB1.T <=20)
        #keep_active = Active * (VS_ST_err < -3) * (BHS.BHS.T_Brine_Out > KB.T) * (KB.T + 3 > KB.T_set)
        keep_active = Active * (VS_ST_err < -1) * (( KB.T_set < KB.T + 3) * (B2.T > KB.T_set))

        #Active = ((VS_ST_err < -3) * (KB.T_set < KB.T - 2) + keep_active > 0)
        Active = ((VS_ST_err < -4)  + keep_active > 0)
        #Active = ((VS_ST_err < -3) + keep_active > 0)
        #Active_withFC = Active * FanCoil_Active #* (Input.T_ambient > 22)
        #Active_withFC = Active * FanCoil_Active * positive(1 - positive(A1.T - T_brine_safe + 5))
        Active_withFC = (VS_ST_err < -3) * FanCoil_Active * positive(1 - positive(T_brine_safe - 5 - A1.T))*Active
                        #+ heating * (FC2.T > B2.T)

        FC.FC.Active_withFC_heating = Heating * (FC1.T > B2.T)

        FC.FC.ActiveFC = Active_withFC
        Active_noFC = Active #* (Input.T_ambient <= 22)

        Heating = (Active <= 0) * (KB.T_set > KB.T + 3) * Enabled

        Free = (Heating <= 0) * (Active <= 0)

        mode_sum = abs(Active) + abs(Heating) + abs(Free)
        a = assertion(mode_sum == 1)

        # Definition af Free mode: Not in heating mode or active mode

        # assert (Free + Active + Heating <= 1)

        Free_or_Heating = (Free + Heating > 0)

        Active_or_Heating = (Active + Heating > 0)

        dT_EMA_Cond = P_EMA_HS_delivered * R_cond
        dT_EMA_Evap = P_EMA_CS_delivered * R_evap

        # Temperature estimations

        # Estimate the EMA evaporation temperature. Cannot be higher than a maximum value, otherwise decided by brine temperature in free cooling and heating mode and by KB set temperature in active mode
        T_EMA_Evap = HP1.T - EMA_dT_Super

        KB.T = HP2.T - KB.P * R_KB
        # Estimate for the EMA condensation temperature. Must be at least EMA evap temperature + constant or VS temperature setpoint. A constant is added for thermal resistance
        cond_T = (EMA_VV <= 0) * VS_ST.T + (EMA_VV > 0) * VV_ST.T

        hot_side_temp = cond_T if non_transcritical else gas_cooler_exit_T

        # T_EMA_Cond = min2(positive(T_EMA_Evap + dT_Evap_Cond, cond_T  + dT_EMA_Cond), T_EMA_Cond_Max)

        T_EMA_Cond = min2(max2(T_EMA_Evap + dT_Evap_Cond, hot_side_temp) + dT_EMA_Cond, T_EMA_Cond_Max)
        T_EMA_Cond_int_dot = T_EMA_Cond
        # Estimate brine input temperature as EMA evaporation temp in heating mode, EMA condensation in active mode or KB setpoint in free cooling only mode
        # BHS_heating = T_EMA_Evap if Heating else (KB.T_set + EMA_dT_Super) * Free
        BHS.BHS.T_Brine_In = A1.T
        FC.FC.T_Brine_In = FC1.T

        VS.T = VS_ST.T - (VS.P - VS.Peak.P)*R_VS
        VV.T = VV_ST.T - (VV.P + VV.P_VVC - VV.Peak.P)*R_VV

    def diff_KB_manifold(self):
        P_KB = min2(-KB.P * positive(1 - positive(-KB.err - 3) * 1), 3e6) - KB.P_Preheat
        # P_KB =positive(min2(-KB.P, -KB.err*1e5))
        # P_KB_int_dot = -KB.err

        # P_KB =

        F_HP = positive(-P_HP / 10000) + 1
        # F_HP = positive(-P_HP / 10000)
        # * (T_EMA_Evap < 20)

        # P_Process_int_dot = (Active-P_Process_int)/(24*3600)

        # P_Process = max2(0, -KB.P_Process)*(KB.T_set >= 20*1)*positive(1 - positive(KB1.T-20)) * (HP_cooling <= 0)

        KB.P_Process = -KB.P_Process_Available

        # P_Process = max2(0, -KB.P_Process_Available) * positive(1 - positive(KB1.T - KB.T_Process)) * (HP_cooling <= 0)

        # P_Process = max2(0, KB.Process_Flow * positive(KB.Process_T_in - dT_Process_Sub - max2(KB.Process_T_out, KB1.T)) * Cp_W)* (HP_cooling <= 0)
        P_Process = min2(-KB.P_Process_Available - P_Process_Sub_ask, max2(0, KB.Process_Flow * positive( KB.Process_T_in - dT_Process_Sub - max2(KB.Process_T_out, KB1.T, A1.T)) * Cp_W)) * (max2(KB1.T, A1.T) <= KB.Process_T_in) * (B2.T <= 10)
        # dT_Process_Sub = P_Process_Sub_ask / Cp_W / KB.Process_Flow

        E_Process_dot = P_Process * J_MWh
        F_Process = abs(P_Process / Flow_Power)
        F_act_max = BHS.BHS.N*2 if BHS.BHS.Ngt0 > 0 else 100
        F_active = min2(F_act_max, positive(P_active_BHS / 30000) + F_Process * Active)

        KB.P_Process_Used = - P_Process

        Separated_Flow = (Heating * (B2.T < F.T) + Active + Free * (B2.T > F.T) > 0)

        # (P_Process_int<0.01)
        # P_Process = positive(min2(-KB.P_Process, -KB.err * 1e5))*(KB.T_set >= 20*1)

        # E_KB_Process_dot = P_Process*0.0000036

        # F_KB = abs(P_KB / 10000) * (Active<=0) + F_HP * Active + P_Process/10000
        F_KB = abs(KB.P / Flow_Power) + 1 + F_Process * (Active <= 0)
        # F_KB = F_KB_ *Active  + max2(F_HP, F_KB_)*Free + min2(F_HP, F_KB_)*Heating

        F_HP_ = positive(-P_HP / Flow_Power) + Heating * positive(B2.T - HP1.T) * 3
        # F_HP__ = max(positive(BHS.BHS.T_Brine_Out - BHS.BHS.T_Brine_In)*2, F_KB + 1, F_HP_)
        F_HP = F_KB * Separated_Flow + (Separated_Flow<= 0) *(min2(F_HP_, F_KB) * Free + max2(F_HP_, F_KB) * Heating)

        # F_BHS = F_active + 1 + abs(F_HP - F_KB) * Heating + abs(F_HP + F_KB) * Free
        F_BHS = max(F_active, abs(BHS.BHS.T_Brine_Out - BHS.BHS.T_Brine_In) * 0.33, .1) * Separated_Flow +(Separated_Flow<= 0) * ((F_HP - F_KB) * Heating + F_KB * Free)
        BHS.BHS.F = F_BHS

        # P_active = 10*1
        # P_BHS = BHS.P
        # P_HP = 5*1

        # Free_BHS = (B2.T < max2(C.T, HP2.T))
        # Active Cooling
        V0 = ((Separated_Flow + Heating) > 0)
        V1 = (V0 <= 0)
        # V2 = (Free + Heating >0)
        V2 = (Separated_Flow <= 0)
        V3 = Separated_Flow

        F1 = F_BHS - F_active
        F2 = F_BHS
        F3 = F_BHS * Separated_Flow
        F4 = F_BHS * V2
        F5 = F_BHS * V2
        F6 = (F_HP - F_KB) * Heating * (Separated_Flow<=0)
        # F7 = F_HP - F_KB * V3 + (-F_BHS - F_KB) * Heating - F_KB* V1

        F7 = (F_HP - F_KB) * Separated_Flow + (F_HP - F_KB) * Free*(Separated_Flow<=0)

        F8 = F_HP - F_KB * V0
        F9 = F_HP
        F10 = F_HP - F_KB
        F11 = F_KB * V1
        F12 = F_KB * V0

        C_ = assertion(abs(F11 + F6 - F4) < 1e-6, Active, Heating)
        D_ = assertion(abs(F10 - F6 - F7) < 1e-6, Active, Heating)
        E_ = assertion(abs(F5 + F7 - F8) < 1e-6, Active, Heating)
        F_ = assertion(abs(F8 + F12 - F9) < 1e-6, Active, Heating)
        HP1_ = assertion(abs(F9 - F_HP) < 1e-6, Active, Heating)
        HP2_ = assertion(abs(F_HP - F_KB - F10) < 1e-6, Active, Heating)
        KB1_ = assertion(abs(F_KB - F11 - F12) < 1e-6, Active, Heating)
        A1_ = assertion(abs(F1 + F_active - F_BHS) < 1e-6, Active, Heating)
        A2_ = assertion(abs(F2 - F1 - F_active) < 1e-6, Active, Heating)
        B1_ = assertion(abs(F3 + F4 - F2) < 1e-6, Active, Heating)
        B2_ = assertion(abs(F_BHS - F3 - F5) < 1e-6, Active, Heating)

        # F1
        # F1_H_dot = F1 * (A2.h if F1 > 0 else A1.h)
        F1_H_dot = mult_sign(F1, A2.h, A1.h)
        F2_H_dot = mult_sign(F2, B1.h, A2.h)
        F3_H_dot = mult_sign(F3, B2.h, B1.h)
        F4_H_dot = mult_sign(F4, C.h, B1.h)
        F5_H_dot = mult_sign(F5, B2.h, E.h)
        F6_H_dot = mult_sign(F6, D.h, C.h)
        F7_H_dot = mult_sign(F7, D.h, E.h)
        F8_H_dot = mult_sign(F8, E.h, F.h)
        F9_H_dot = mult_sign(F9, F.h, HP1.h)
        F10_H_dot = mult_sign(F10, HP2.h, D.h)
        F11_H_dot = mult_sign(F11, KB1.h, C.h)
        F12_H_dot = mult_sign(F12, KB1.h, F.h)
        Active_H_dot = A2.h * F_active
        BHS_H_dot = A1.h * F_BHS
        KB_H_dot = HP2.h * F_KB
        HP_H_dot = HP1.h * F_HP

        P_Process_KB = P_Process * Heating
        P_Process_BHS = P_Process * (Heating <= 0)*BHS.BHS.Ngt0

        A1.H_dot = -BHS_H_dot + F1_H_dot + P_active_BHS+ Active_H_dot + P_Process_BHS
        A2.H_dot = -F1_H_dot - Active_H_dot + F2_H_dot
        B1.H_dot = -F2_H_dot + F4_H_dot + F3_H_dot
        B2.H_dot = BHS_H_dot - BHS.BHS.P - F3_H_dot - F5_H_dot +BHS.BHS.P_pump
        E.H_dot = F5_H_dot + F7_H_dot - F8_H_dot
        F.H_dot = F8_H_dot + F12_H_dot - F9_H_dot
        HP1.H_dot = F9_H_dot - HP_H_dot
        HP2.H_dot = HP_H_dot + P_HP - KB_H_dot - F10_H_dot
        D.H_dot = F10_H_dot - F6_H_dot - F7_H_dot
        C.H_dot = F11_H_dot + F6_H_dot - F4_H_dot
        KB1.H_dot = KB_H_dot + P_KB - F12_H_dot - F11_H_dot + P_Process_KB - P_heating_FC
        # + KB.Peak.P

    def diff_EMA(self):
        # Get EMA heating from Bitzer polynomials
        Enabled_N_comp_EMA = Enabled * N_comp_EMA
        # EMA_Safe =(T_EMA_Cond <= 62)*(T_EMA_Evap < 20)

        EMA_Safe = (HP2.T > -20)

        # T_EMA_Evap = 20*1
        # T_EMA_Cond = 40*1
        P_EMA_m_dot = calc_poly2x3(EMA_m_dot_poly, T_EMA_Evap, T_EMA_Cond) / 3600 * Enabled_N_comp_EMA * EMA_Safe
        P_EMA_Heat_ = calc_poly2x3(EMA_Q_poly, T_EMA_Evap, T_EMA_Cond) * Enabled_N_comp_EMA * EMA_Safe
        # Get EMA electricity from Bitzer polynomials
        P_EMA_Elec = calc_poly2x3(EMA_power_poly, T_EMA_Evap, T_EMA_Cond) * Enabled_N_comp_EMA * X_Pumps * X_Regulation * EMA_Safe
        P_EMA_Elec_Heat_Added = P_EMA_Elec * eta_thermal
        P_EMA_Heat = P_EMA_Heat_ + P_EMA_Elec_Heat_Added

        COP_EMA = no_zero_div(P_EMA_Heat_, P_EMA_Elec, 0) + 1

        P_EMA_cool_equi = P_EMA_Heat_
        # positive(P_EMA_Heat - P_EMA_Elec_Heat_Added)

    def diff_EMB(self):

        VV_ST_T_dT_EMB_Cond = VV_ST.T + dT_EMB_Cond
        T_EMB_Cond = VV_ST_T_dT_EMB_Cond if P_EMB_VV > 0 else VS_ST.T + dT_EMB_Cond
        T_EMB2_Cond = VS_ST.T + dT_EMB_Cond

        N_comp_EMB_Enabled = N_comp_EMB * Enabled
        X_Pumps_X_Regulation = X_Pumps * X_Regulation

        P_EMB_Heat_ = calc_poly2x3(EMB_Q_poly, T_EMB_Evap, T_EMB_Cond) * N_comp_EMB_Enabled
        P_EMB_Elec_ = calc_poly2x3(EMB_power_poly, T_EMB_Evap, T_EMB_Cond) * N_comp_EMB_Enabled * X_Pumps_X_Regulation
        P_EMB_m_dot = calc_poly2x3(EMB_m_dot_poly, T_EMB_Evap, T_EMB_Cond) / 3600 * N_comp_EMB_Enabled

        P_EMB_Elec_Heat_Added = P_EMB_Elec_ * eta_thermal

        P_EMB_Heat = P_EMB_Elec_Heat_Added + P_EMB_Heat_

        P_EMB_HS_max = P_EMB_Heat

        N_comp_EMB2_Enabled = N_comp_EMB2 * Enabled

        P_EMB2_Heat_ = calc_poly2x3(EMB_Q_poly, T_EMB2_Evap, T_EMB2_Cond) * N_comp_EMB2_Enabled
        P_EMB2_Elec_ = calc_poly2x3(EMB_power_poly, T_EMB2_Evap,T_EMB2_Cond) * N_comp_EMB2_Enabled * X_Pumps_X_Regulation
        P_EMB2_m_dot = calc_poly2x3(EMB_m_dot_poly, T_EMB2_Evap, T_EMB2_Cond) / 3600 * N_comp_EMB2_Enabled

        P_EMB2_Elec_Heat_Added = P_EMB2_Elec_ * eta_thermal

        P_EMB2_Heat = P_EMB2_Elec_Heat_Added + P_EMB2_Heat_

        P_EMB2_HS_max = P_EMB2_Heat

        # COP_EMB_div = P_EMB_Elec_ if P_EMB_Elec_ > 0 else 1e10
        COP_EMB = no_zero_div(P_EMB_Heat_, P_EMB_Elec_, 0) + 1

        # COP_EMB2_div = P_EMB2_Elec_ if P_EMB2_Elec_ > 0 else 1e10
        COP_EMB2 = no_zero_div(P_EMB2_Heat_, P_EMB2_Elec_, 0) + 1

        # EMB_energy_dot = P_EMB_VV

    def diff_w_sub_process(self):
        # P_Process_Sub_ask = positive(min2(positive(1e5 * (20 * 1 - SCC2.T), -KB.P_Process - P_Process))) * Sub_Process_Heat
        P_Process_Sub_ask = KB.Process_Flow * positive(KB.Process_T_in - max2(SCC1.T, KB.Process_T_out)) * Cp_W * (SCC2.T <= T_EMB_Evap_Max + EMB_dT_Super)
        dT_Process_Sub = no_zero_div(P_Process_Sub_ask / Cp_W, KB.Process_Flow, 0)

    def diff_w_subcooling(self):
        gas_cooler_exit_T = SCC3.T


        T_EMB_Evap = SCC2.T - EMB_dT_Super
        # T_EMB_Evap = 20
        T_EMB2_Evap = SCC3.T - EMB_dT_Super
        # T_EMB2_Evap = 20

        P_EMA_Sub_Cond_ask = positive(1e5 * (EMA_Cond_SCC_min - SCC2.T))
      #  P_EMA_Sub = positive((T_EMA_Cond - 5 - SCC3.T) * Cp_EMA * P_EMA_m_dot * P_EMA_HS_delivered / (1 + P_EMA_Heat)) * (SCC2.T <= T_EMB_Evap_Max + EMB_dT_Super)

        #temperature of refrigerant at subcooler outlet: dt approach set to 5 K
        T_EMA_OutSC = 5+SCC3.T
        # saturation pressure at condensation temperature:
        p_EMA_Cond = interp_bilin(EMA_p_Tq,  T_EMA_Cond, 0)
        #p_EMA_Cond = ME.bilinear_interpolation(0, T_EMA_Cond, 0 , [0], EMA_p_Tq)

        #calculate respective enthalpies from property tables:
        h_EMA_OutSC = interp_bilin(EMA_h_Tp, T_EMA_OutSC, ln(p_EMA_Cond))
        h_EMA_Cond  = interp_bilin(EMA_h_Tp, T_EMA_Cond, ln(p_EMA_Cond))
        #h_EMA_OutSC = ME.bilinear_interpolation(0, T_EMA_OutSC, ln(p_EMA_Cond), [0], EMA_h_Tp)
        #h_EMA_Cond  = ME.bilinear_interpolation(0, T_EMA_Cond , ln(p_EMA_Cond), [0], EMA_h_Tp)

        # heat transfer estimated by enthalpy change and not cp dT
        P_EMA_Sub = positive((h_EMA_Cond - h_EMA_OutSC) * P_EMA_m_dot * P_EMA_HS_delivered / (1 + P_EMA_Heat)) * (SCC2.T <= T_EMB_Evap_Max + EMB_dT_Super)
        # *(SCC2.T <= VS_SCC_min)

        EMA_Sub_Flow = max2(min2((P_EMA_Sub + P_EMA_Sub_Cond_delivered) / EMA_Sub_Flow_P_div, N_comp_EMA / 2), 0.1)

        # SCC.H_dot =  +   - P_Sub_EMHW12

        # X_EMB_VV = (VV.T_set - SCC2.T)/(VV.T_set - T_Cold)
        P_EMB_VV = positive(min2(P_EMB_HS_max, X_VV_HP * P_EMB_Heat) * (SCC2.T > T_SCC_VV_Min) * VV_Prod_EMB)

        P_EMB1_VS = (SCC2.T > VS_SCC2_min) * VS_Prod_EMB * (P_EMB_VV <= 0) * positive(min2(P_EMB_HS_max, (VS_ST_err - 1) * 1e6)) * (Active <= 0)

        P_EMB2_VS = (SCC3.T > VS_SCC3_min) * VS_Prod_EMB2 * positive(min2(P_EMB2_HS_max, (VS_ST_err - 1) * 1e6)) * (Active <= 0)

        # P_EMB2_VS = positive(min2(P_EMB2_HS_max, VS_ST_err* 1e6)  * VS_Prod_EMB2)*(Active<=0)

        # P_EMB1_VS = positive(min2((P_EMB_VV <= 0) * P_EMB_HS_max, VS_ST_err * 1e6)* VS_Prod_EMB)*(Active<=0)

        P_EMB_VS = P_EMB1_VS + P_EMB2_VS

        dT_VV = VV.T_set - T_Cold
        # VV_Err = -VV.P*(VV.T_set>VV.T)
        # P_Sub_EMHW12 = VV.P * (1 - X_EMB_VV) * VV_Prod
        P_Sub_EMHW1 = min2(positive(VV_Prod_EMHW1 * (SCC3.T - T_Cold) / dT_VV), 1) * VV.P * Enabled
        P_Sub_EMHW2 = min2(positive(VV_Prod_EMHW2 * (SCC1.T - SCC3.T) / dT_VV), 1) * (VV.P - P_Sub_EMHW1) * Enabled

        # EMA_Sub_Flow_m_SCC = EMA_Sub_Flow/m_SCC
        Process_Sub_Flow = P_Process_Sub_ask / Cp_W / 5
        EMA_Sub_Flow = EMA_Sub_Flow + Process_Sub_Flow
        SCC1.H_dot = (SCC3.h - SCC1.h) * EMA_Sub_Flow + P_EMA_Sub_Cond_delivered + P_EMA_Sub - P_Sub_EMHW2 + P_Process_Sub_ask
        SCC2.H_dot = (SCC1.h - SCC2.h) * EMA_Sub_Flow - P_Sub_EMB
        SCC3.H_dot = (SCC2.h - SCC3.h) * EMA_Sub_Flow - P_Sub_EMHW1 - P_Sub_EMB2

        E_Process_Sub_dot = P_Process_Sub_ask * J_MWh

    def diff_wo_subcooling(self):
        gas_cooler_exit_T = (EMA_VV <= 0) * VS_ST_Return.T + (EMA_VV > 0) * VV_ST.T

        # X_EMB_VV = 1
        T_EMB_Evap = KB.T - EMB_dT_Super
        T_EMB2_Evap = KB.T - EMB_dT_Super

        P_EMB_VV = positive(min2(P_EMB_HS_max, X_VV_HP * P_EMB_Heat) * VV_Prod_EMB)

        P_EMB1_VS = positive(min2((P_EMB_VV <= 0) * P_EMB_HS_max, VS_ST_err * 1e6) * VS_Prod_EMB) * (Active <= 0)

        P_EMB2_VS = positive(min2(P_EMB2_HS_max, VS_ST_err * 1e6) * VS_Prod_EMB2) * (Active <= 0)

        P_EMB_VS = P_EMB1_VS + P_EMB2_VS

        P_EMA_Sub_Cond_ask = 0
        P_EMA_Sub = 0
        P_KB_EMB = P_Sub_EMB + P_Sub_EMB2

    def diff_VS_VV_preheat(self):

        dT_VV = VV.T_set - T_Cold
        P_VS_VV_Preheat = min2(positive(VS_VV_Preheat * (VS_ST.T - T_Cold) / dT_VV), 1) * VV.P * Enabled

    def diff(self):
        # temperature of refrigerant at subcooler outlet: dt approach set to 3 K
        T_EMB_OutSC  = 3+VS_ST_Return.T # degC
        T_EMB2_OutSC = 3 + VS_ST_Return.T  # degC

        # saturation pressure at condensation temperature:
        p_EMB_Cond  = interp_bilin(EMB_p_Tq, T_EMB_Cond, 0)
        p_EMB2_Cond = interp_bilin(EMB_p_Tq, T_EMB2_Cond, 0)

        #p_EMB_Cond = ME.bilinear_interpolation(0, T_EMB_Cond, 0, [0], EMB_p_Tq)
        #p_EMB2_Cond = ME.bilinear_interpolation(0, T_EMB2_Cond, 0, [0], EMB_p_Tq)

        h_EMB_Cond  = interp_bilin(EMB_h_Tp, T_EMB_Cond, ln(p_EMB_Cond))
        h_EMB2_Cond = interp_bilin(EMB_h_Tp, T_EMB2_Cond, ln(p_EMB2_Cond))

        h_EMB_OutSC  = interp_bilin(EMB_h_Tp,T_EMB_OutSC, ln(p_EMB_Cond))
        h_EMB2_OutSC = interp_bilin(EMB_h_Tp,T_EMB2_OutSC, ln(p_EMB2_Cond))

        P_EMB_Sub = no_zero_div(positive((T_EMB_Cond - 3 - VS_ST_Return.T)) * P_EMB_m_dot * Cp_EMB * (P_EMB_VS + P_EMB_VV), P_EMB_HS_max, 0)*(VS_ST_err>0)
        #calculate respective enthalpies from property tables:
        #h_EMB_OutSC = ME.bilinear_interpolation(0, T_EMB_OutSC, ln(p_EMB_Cond), [0], EMB_h_Tp)
        #h_EMB2_OutSC = ME.bilinear_interpolation(0, T_EMB2_OutSC, ln(p_EMB2_Cond), [0], EMB_h_Tp)
        #h_EMB_Cond  = ME.bilinear_interpolation(0, T_EMB_Cond , ln(p_EMB_Cond), [0], EMB_h_Tp)
        #h_EMB2_Cond = ME.bilinear_interpolation(0, T_EMB2_Cond, ln(p_EMB2_Cond), [0], EMB_h_Tp)

        P_EMB2_Sub = no_zero_div(positive((T_EMB2_Cond - 3 - VS_ST_Return.T)) * P_EMB2_m_dot * Cp_EMB * P_EMB2_VS,P_EMB2_HS_max, 0)*(VS_ST_err>0)
        P_EMB_Sub = no_zero_div(positive((h_EMB_Cond - h_EMB_OutSC)) * P_EMB_m_dot * (P_EMB_VS + P_EMB_VV), P_EMB_HS_max, 0)
        P_EMB2_Sub = no_zero_div(positive((h_EMB2_Cond - h_EMB2_OutSC)) * P_EMB2_m_dot * P_EMB2_VS, P_EMB2_HS_max, 0)

        #P_EMB_Sub = no_zero_div(positive((T_EMB_Cond - 3 - VS_ST_Return.T)) * P_EMB_m_dot * Cp_EMB * (P_EMB_VS + P_EMB_VV), P_EMB_HS_max, 0)

        #P_EMB2_Sub = no_zero_div(positive((T_EMB2_Cond - 3 - VS_ST_Return.T)) * P_EMB2_m_dot * Cp_EMB * P_EMB2_VS,P_EMB2_HS_max, 0)

        P_EMB1_VS = P_EMB_Sub + P_EMB1_VS
        P_EMB2_VS = P_EMB2_Sub + P_EMB2_VS

        VV_ST_err = (VV.T_set + VV_dT_offset) - VV_ST.T

        X_VV_HP_start = (VV_ST_err - 2 > 0) * Enabled
        X_VV_HP_keep = (VV_ST_err + 2 > 0) * Enabled

        X_VV_HP = (X_VV_HP_start + X_VV_HP_keep * (X_VV_HP > 0) > 0)

        EMA_VV_start = (VV_ST_err > Err_EMA_VV_start) * VV_Prod_EMA
        EMA_VV_keep = (VV_ST_err > Err_EMA_VV_keep) * (EMA_VV > 0) * VV_Prod_EMA

        EMA_VV = X_VV_HP * ((EMA_VV_start + EMA_VV_keep) > 0) * VV_Prod_EMA

        # EMA VS
        VS_T_set = max2(T_VS_min, (VS.T_set + VS_dT_offset))
        VS_ST_err = VS_T_set - VS_ST.T

        X_VS_HP = positive(VS_ST_err * 10) * Enabled

        EMA_VS_start = (VS_ST_err > Err_EMA_VS_start)
        EMA_VS_keep = (VS_ST_err > Err_EMA_VS_keep) * (EMA_VS > 0)

        EMA_VS = ((EMA_VS_start + EMA_VS_keep) > 0)

        # EMA KB Cooling
        KB_ST_err = -KB.err
        # + KB_dT_offset)

        X_KB_HP = positive(KB_ST_err) * Enabled

        EMA_KB_start = (KB_ST_err > Err_EMA_KB_start)
        EMA_KB_keep = (KB_ST_err > Err_EMA_KB_keep) * (EMA_KB > 0)

        EMA_KB = ((EMA_KB_start + EMA_KB_keep) > 0)

        # T_brine_safe = positive(1 - positive(BHS.BHS.T_Brine_Out - T_max_brine))*positive(1 - positive(A1.T - T_max_brine))
        T_brine_safe = positive(1 - positive(A1.T - T_max_brine))

        P_EMA_VV_ask = EMA_VV * P_EMA_Heat * (T_EMA_Cond >= VV_ST.T)
        P_EMA_VS_ask = EMA_VS * P_EMA_Heat * (T_EMA_Cond >= VS_ST.T)
        P_EMA_KB_ask = EMA_KB * P_EMA_cool_equi * (positive(T_max_brine-1 - A1.T) + (VS_ST_err > -5) > 0)
        P_EMA_HS_ask = min2(350e6, max2(P_EMA_KB_ask, P_EMA_VS_ask + P_EMA_VV_ask + P_EMA_Sub_Cond_ask))
                            #* (VS_ST.T < max2(VS_T_set + 5, A2.T + 10)))

        #P_EMA_HS_ask_dot = (P_EMA_HS_ask_ - P_EMA_HS_ask)*0.01

        P_EMA_HS_delivered = min2(P_EMA_HS_ask, P_EMA_Heat) * (VS_ST.T <= T_EMA_Cond_Max)

        P_EMA_Elec_act = P_EMA_HS_delivered / COP_EMA

        P_EMA_CS_delivered = -P_EMA_HS_delivered + P_EMA_Elec_act * eta_thermal - P_EMA_Sub

        # P_EMA_Sub_Cond_delivered = min2(P_EMA_HS_delivered, P_EMA_Sub_Cond_ask)
        P_EMA_Sub_Cond_delivered = min2(P_EMA_Sub_Cond_ask, P_EMA_HS_delivered)

        P_EMA_VV_VS_delivered = P_EMA_HS_delivered - P_EMA_Sub_Cond_delivered

        P_EMA_VV_delivered = min2(P_EMA_VV_VS_delivered, P_EMA_VV_ask)

        P_EMA_VS_delivered = P_EMA_VV_VS_delivered - P_EMA_VV_delivered

        # EMB

        P_EMB1_Elec_act = (P_EMB_VV + P_EMB1_VS) / COP_EMB
        P_EMB2_Elec_act = P_EMB2_VS / COP_EMB

        P_Sub_EMB = positive(P_EMB1_VS + P_EMB_VV - P_EMB1_Elec_act * eta_thermal)

        P_Sub_EMB2 = positive(P_EMB2_VS - P_EMB2_Elec_act * eta_thermal)

        # Setting H dot values
        # VS

        ##todo: Shouldn't be VS_ST_return?? (see ControlMachines Frolunda Torg)
        P_active_BHS = min2(3e6, (Active > 0) * positive(VS_ST.T - A2.T) * 1e6 * T_brine_safe * positive(VS_ST.T - VS_T_set - 3) * (A1.T < T_max_brine))*BHS.BHS.Ngt0

        # Calculate FC power
        #P_active_FC = max2((Active_withFC > 0) * positive(VS_ST_Return.T - FC2.T) * 1e5 * positive(VS_ST.T - VS_T_set - 3),FC.FC.P_cooling_nominal) * (FC1.T < T_max_brine)
        P_active_FC_ = min2(3e6, (Active_withFC > 0) * positive(VS_ST.T - FC2.T) * 1e6/2 * positive(VS_ST.T - VS_T_set - 3))
        P_active_FC_dot = (P_active_FC_ - P_active_FC) * 0.01

        P_heating_FC_ = -min2(3e6, (FC.FC.Active_withFC_heating > 0) * positive(FC2.T - B2.T) * 1e6/2)*Heating
        P_heating_FC_dot = (P_heating_FC_ - P_heating_FC) * 0.01

        #* (FC1.T < T_max_brine)
        F_active_FC = max2(0, abs(FC1.T - FC2.T) * 10)
        F_FC = max2(0.1, F_active_FC)
        P_active = P_active_BHS + P_active_FC

        # Adjust the flow to minimize the temperature difference
        #F_active_FC = max2(P_active_FC/10000, abs(FC1.T - FC2.T) * 10)

        # Add a bit of flow not to have heat building up
        F_FC = max2(0.1, F_active_FC)
        # F_FC = min(F_active_FC, FC.FC.F_brine_nominal)
        F13 = F_active_FC - F_FC

        # If flow becomes negative we need specific enthalpy of FC2
        F13_H_dot = mult_sign(F13, FC1.h, FC2.h)
        FC1.H_dot = -F_FC * FC1.h - F13_H_dot + F_active_FC * FC2.h + P_active_FC + P_heating_FC
        FC2.H_dot = F_FC * FC1.h + F13_H_dot - F_active_FC * FC2.h - FC.FC.P
                    #+ FC.FC.P_el_fan

        # transfer temperature and flow back to FC model
        FC.FC.F = F_FC


        P_VS = (VS.P - KB.P_Preheat) * positive(1 - positive(VS.err - 3))

        VS.Peak.P = VS.P - P_VS - KB.P_Preheat

        F_VS_ = no_zero_div(VS.P, (VS.T_set - VS.T_Return), 10) / Cp_W if VS.T_Return > 0 else abs(P_VS / 30000)
        F_VS = max2(10, min2(50, F_VS_ + abs(P_active / 10000)))

        VS_ST_H_dot = F_VS * (VS_ST.h - VS_ST_Return.h)

        VS_ST.H_dot = P_EMA_VS_delivered + P_EMB1_VS + P_EMB2_VS - VS_ST_H_dot  # - P_EMA_Sub_Cond_delivered

        VS_ST_Return.H_dot = VS_ST_H_dot - P_active - P_VS - P_VS_VV_Preheat

        VV_P_rem = VV.P + VV.P_VVC - P_Sub_EMHW1 - P_Sub_EMHW2 - P_VS_VV_Preheat

        E_VS_VV_Preheat_dot = P_VS_VV_Preheat * J_MWh
        # P_VV_ST = -VV.P +  - VV.P_VVC

        P_VV = VV_P_rem * positive(1 - positive(VV.err - 3))

        VV.Peak.P = VV_P_rem - P_VV

        # VV
        VV_ST.H_dot = P_EMA_VV_delivered + P_EMB_VV - P_VV

        # VV.Peak.P = positive((VV.T_set - VV_ST.T - 3) * VV.P * 0.33)

        # KB
        P_HP = P_EMA_CS_delivered - P_KB_EMB

        KB.Peak.P = KB.P + P_KB
        # -positive(HP2.T - KB.T_set - 1) * 1e5

        # P_KB = KB.P

        # P_BHS_Heating = (P_EMA_CS - KB_P)*Heating - P_KB_EMB
        P_BHS = (BHS.BHS.T_Brine_Out - BHS.BHS.T_Brine_In) * abs(F_BHS) * Cp_W
        BHS.BHS.P_ = -P_BHS

        BHS_E_Free_dot = -BHS.BHS.P * J_MWh * Free
        BHS_E_Active_dot = -BHS.BHS.P * J_MWh * Active
        BHS_E_Heating_dot = -BHS.BHS.P * J_MWh * Heating

        FC_E_Active_dot = FC.FC.P * J_MWh * Active_withFC

        P_Comp = (P_EMA_Elec_act + P_EMB1_Elec_act + P_EMB2_Elec_act)
        COP_EM = no_zero_div(P_EMA_HS_delivered + P_EMB1_VS + P_EMB_VV + P_EMB2_VS, P_Comp, 0)
        P_Elec = -P_Comp
        Energy_Electrical_Loss_dot = P_Comp * (1 - eta_thermal) * J_MWh
        Energy_Consumption_dot = (P_Comp + BHS.BHS.P_pump + 0*FC.FC.P_el_fan) * J_MWh
        Energy_Consumption_Compressors_dot = P_Comp / X_Pumps * J_MWh
        Energy_Consumption_Pumps_dot = (1 - 1 / X_Pumps) * P_Comp * J_MWh
        Energy_Consumption_FanCoil_dot = FC.FC.P_el_fan * J_MWh

        E_EMB1_VS_dot = P_EMB1_VS * J_MWh
        E_EMB2_VS_dot = P_EMB2_VS * J_MWh
        E_EMB_VV_dot = P_EMB_VV * J_MWh
        E_EMA_VS_delivered_dot = P_EMA_VS_delivered * J_MWh

