import sim_components.generic.items as B
import numpy as np

import sim_tools.assemble_solve.model_solver as ME
import pickle
import math

class Fluid_Props(B.Item):
    def construct_Item(self,**c):
        self.props = pickle.load( open('thermodynamics/Interp/'+c['file'], "rb" ))
        
        print(self.props['lim'])
        
        def sa(s,k,v,t):
            setattr(s, k+'_'+t, v[t])

        for k,v in self.props['data'].items():
            
            sa(self,k,v,'T')
            sa(self,k,v,'p')
            sa(self,k,v,'h')
            sa(self,k,v,'s')
            sa(self,k,v,'q')
            sa(self,k,v,'rho')
            
class Liquid_Props(B.Item):
    def construct_Item(self,**c):
        self.rho=c['rho']
        self.Cp=c['Cp']
        self.T=c['T']
        self.v_dyn=c['v_dyn']
        self.v_kin=c['v_kin']
        self.liq_name=c['liquid']
        
        self.v_dyn_lin=np.concatenate([[len(self.T),min(self.T),max(self.T)],self.T,self.v_dyn])
        #self.v_kin_lin=np.concatenate([[len(self.T),min(self.T),max(self.T)],self.T,self.v_kin])

        self.add_var('Cp', val=self.Cp)

class Reservoir(B.Node):
    def construct_Node(self,Fluid,q0=1, T0=20,  dT_Super=0, p0=1e3):
        
        self.Fluid=Fluid
        
        if isinstance(Fluid,Fluid_Props):
            #print()
            print(self.name)

            p=ME.bilinear_interpolation(0, T0, q0, [0], self.Fluid.Tq_p)

            h=ME.bilinear_interpolation(0, T0+dT_Super, np.log(p), [0], self.Fluid.Tp_h)
            rho=ME.bilinear_interpolation(0, T0+dT_Super, np.log(p), [0], self.Fluid.Tp_rho)
            self.Variables=[
                        {'name': 'm', 'unit': 'kg', 'type':'reservoir', 'val':rho},
                        {'name': 'H', 'unit': 'J', 'type':'reservoir', 'val':h*rho},
                        {'name': 'h', 'unit': 'J', 'type':'reservoir', 'val':h},
                        {'name': 'p', 'unit': 'kPa', 'type':'reservoir', 'val':p},
                        {'name': 'T', 'unit': 'K', 'type':'reservoir', 'val':T0},
                        {'name': 'V', 'unit': 'm3', 'type':'param', 'val':.1}
                        ]

            print('p: ',p)
            print('rho: ',rho)
            print('h: ', h)
        
        elif isinstance(Fluid,Liquid_Props):
            
            m0=self.Fluid.rho
            
            self.calc_funcs=[self.Liquid_reservoir_update]
            
            self.Variables=[
                        {'name': 'm', 'unit': 'kg', 'type':'reservoir', 'val':m0},
                        {'name': 'H', 'unit': 'J', 'type':'reservoir', 'val': self.Fluid.Cp*(T0+273.15)*m0},
                        {'name': 'h', 'unit': 'J', 'type':'reservoir', 'val': self.Fluid.Cp*(T0+273.15)},
                        {'name': 'p', 'unit': 'kPa', 'type':'reservoir', 'val':p0},
                        {'name': 'T', 'unit': 'K', 'type':'reservoir', 'val':T0},
                        {'name': 'V', 'unit': 'm3', 'type':'param', 'val':1},
                        {'name': 'C', 'unit': 'J/K', 'type':'param', 'val':m0*self.Fluid.Cp},
                        {'name': 'Cp', 'unit': 'J/K', 'type':'param', 'val':self.Fluid.Cp},
                        {'name': 'T_Set', 'unit': 'K', 'type':'param', 'val':T0}
                        ]
        else:
            raise ValueError('The medium class not recognized!')
    
    def Liquid_reservoir_update(self):
        T=T_Set
        H=(T_Set+273.15)*C
        h=(T_Set+273.15)*Cp

class ThermalReservoir(B.Node):
    def construct_Node(self, Fluid, V, T0=20):

        self.Fluid = Fluid


        assert isinstance(Fluid, Liquid_Props), 'must be liquid!'

        self.calc_funcs = [self.Liquid_Volume_diff_f]

        self.Liquid = Fluid

        m0 = self.Fluid.rho * V * 1.1

        self.Variables = [
            {'name': 'm', 'unit': 'kg', 'type': 'state', 'val': m0},
            {'name': 'H', 'unit': 'J', 'type': 'state', 'val': 0},
            #{'name': 'H_dot', 'unit': 'J/s', 'type': 'param', 'val': 0},
            #{'name': 'E', 'unit': 'J', 'type': 'param', 'val': 0},
            {'name': 'h', 'unit': 'J', 'type': 'param', 'val': self.Fluid.Cp * (T0 + 273.15)},
            {'name': 'p', 'unit': 'kPa', 'type': 'state', 'val': 1e3},
            {'name': 'T', 'unit': 'K', 'type': 'calc', 'val': T0},
            {'name': 'T_set', 'unit': 'K', 'type': 'param', 'val': T0},
            {'name': 'V', 'unit': 'm3', 'type': 'param', 'val': V},
            {'name': 'Cp', 'unit': 'J/K', 'type': 'param', 'val': self.Fluid.Cp},
            {'name': 'rho0', 'unit': 'kg/m3', 'type': 'param', 'val': self.Fluid.rho},
            {'name': 'kP', 'unit': 'Hz', 'type': 'param', 'val': .1}
        ]

    def Liquid_Volume_diff_f(self):
        T = T_set
        h = Cp * (T_set + 273.15)
        #H = m * h




        d_rho = (m / V - rho0) / rho0
        rp = d_rho * 1e4 + 1e3
        p_dot = (rp - p) * kP



class Volume(B.Node):
    def construct_Node(self, Fluid=None, V=0.1, q0=0, T0=20, p0=0):

        self.Fluid=Fluid
        
        #asdasd=dfsdf
        if isinstance(Fluid,Fluid_Props):
            
            self.rh_p=self.Fluid.rh_p
            self.rh_T=self.Fluid.rh_T
            self.rh_q = self.Fluid.rh_q
            
            #[rho, h, p]=self.Fluid.calc_props('Tq',[T0,q0],'rhp')
            
            rho=ME.bilinear_interpolation(0, T0, q0, [0], self.Fluid.Tq_rho)
            p=ME.bilinear_interpolation(0, T0, q0, [0], self.Fluid.Tq_p)
            h=ME.bilinear_interpolation(0, T0, q0, [0], self.Fluid.Tq_h)
            
            # print('vol')
            # print(T0)
            # print(q0)
            # print(p)
            # print(h)
            # print(rho)
            #asdads=fdsfdsdf
            m=rho*V
            H=h*m
            #print(self.name)
            #print('p: ', p)
            #print('rho: ', rho)
            #print('m: ', m)
            #print('h: ', h)

            self.calc_funcs=[self.F2P_Volume_diff_f]
            
            self.Variables=[
                        {'name': 'm', 'unit': 'kg', 'type':'state', 'val':m},
                        {'name': 'H', 'unit': 'J', 'type':'state', 'val':H},
                        {'name': 'h', 'unit': 'J/kg', 'type':'param', 'val':h},
                        {'name': 'p', 'unit': 'kPa', 'type':'state', 'val':p},
                        {'name': 'T', 'unit': 'K', 'type':'state', 'val':T0},
                        {'name': 'V', 'unit': 'm3', 'type':'param', 'val':V},
                        {'name': 'kP', 'unit': 'Hz', 'type':'param', 'val':.1}
                        ]
        
        elif isinstance(Fluid,Liquid_Props):
            
            self.calc_funcs=[self.Liquid_Volume_diff_f]
            
            self.Liquid=Fluid
            
            m0=self.Fluid.rho*V*1.1
        
            self.Variables=[
                        {'name': 'm', 'unit': 'kg', 'type':'state', 'val':m0},
                        {'name': 'H', 'unit': 'J', 'type':'state', 'val': self.Fluid.Cp*(T0+273.15)*m0},
                        {'name': 'h', 'unit': 'J', 'type':'param', 'val': self.Fluid.Cp*(T0+273.15)},
                        {'name': 'q', 'unit': '1', 'type': 'param', 'val': 1},
                        {'name': 'p', 'unit': 'kPa', 'type':'state', 'val':1e3},
                        {'name': 'T', 'unit': 'K', 'type':'state', 'val':T0},
                        {'name': 'V', 'unit': 'm3', 'type':'param', 'val':V},
                        {'name': 'Cp', 'unit': 'J/K', 'type':'param', 'val':self.Fluid.Cp},
                        {'name': 'rho0', 'unit': 'kg/m3', 'type':'param', 'val':self.Fluid.rho},
                        {'name': 'kP', 'unit': 'Hz', 'type':'param', 'val':.1}
                        ]
        else:
            print(Fluid)
            raise ValueError('The medium class not recognized!')
        
    def F2P_Volume_diff_f(self):
        #rp=exp(interp_bilin(rh_p, m/V, H/m))
        rho=m/V
        h=H/m

        q=interp_bilin(rh_q, m/V, H/m)
        rp=interp_bilin(rh_p, m/V, H/m)
        rT=interp_bilin(rh_T, m/V, H/m)

        p_dot=(rp-p)*kP
        T_dot=(rT-T)*kP
       
    
        
    def Liquid_Volume_diff_f(self):
        rT=H/Cp/m-273.15
        h=H/m

        d_rho=(m/V-rho0)/rho0
        rp=d_rho*1e4+1e3
        
        p_dot=(rp-p)*kP
        T_dot=(rT-T)*kP
        
      
        
class Pipe(B.Link):
    def construct_Link(self,kV, **c):
        self.side1=c['side1']
        self.side2=c['side2']
        
        self.calc_funcs=[self.Pipe_diff_f]

        self.Variables=[

                    {'name': 'kV', 'unit': 'kg/s/sqrt(kPa)', 'type':'param', 'val':kV},
                    {'name': 'm_dot_max', 'unit': 'kg/s', 'type': 'param', 'val': 100},
                    ]

    def Pipe_diff_f(self):
        T_pipe = ((m_dot > 0) * side1.T + (m_dot <= 0) * side2.T)
        dp = side1.p - side2.p
        m_dot_raw = sign(dp) * sqrt(abs(dp)) * kV
        m_dot = m_dot_raw * (abs(m_dot_raw) <= m_dot_max) + m_dot_max * (abs(m_dot_raw) > m_dot_max) * sign(m_dot_raw)

        #H_dot = m_dot * ((m_dot > 0) * side1.H / side1.m + (m_dot < 0) * side2.H / side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)

        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot

    def Pipe_diff_f2(self):
        T_pipe=((m_dot>0)*side1.T+(m_dot<=0)*side2.T)
        dp=side1.p-side2.p
        m_dot=sign(dp)*sqrt(abs(dp))*kV

        #H_dot=m_dot*((m_dot>0)*side1.H/side1.m+(m_dot<0)*side2.H/side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)

        side1.m_dot=-m_dot
        side2.m_dot=m_dot
        side1.H_dot=-H_dot
        side2.H_dot=H_dot

def make_pipe_lines(host, pipe_lines, kV=5):
    for pl in pipe_lines:

        c_last = None
        for i, c in enumerate(pl):
            if i > 0:
                pipe_tag='Pipe_' + c_last.name + '_' + c.name
                #print(pipe_tag)
                host.add(pipe_tag, Pipe, kV=kV,
                         side1=c_last, side2=c)
            c_last = c

def make_pipe_check_lines(host, pipe_lines, kV=5):
    for pl in pipe_lines:
        print([i.path for i in pl])
        c_last = None
        for i, c in enumerate(pl):
            if i > 0:
                pipe_tag = 'Pipe_Check_' + c_last.name + '_' + c.name
                print(pipe_tag)
                host.add(pipe_tag, Pipe_Check, kV=kV,
                         inlet=c_last, outlet=c)
            c_last = c

def make_network(host, pipe_lines, Fluid, kV=5, T0=20, V=0.1):

    all_volumes = {}
    for pl in pipe_lines:
        for v in pl:
            if isinstance(v, str):
                kwargs = {'Fluid': Fluid, 'T0': T0, 'V': V}
                all_volumes[v] = kwargs
            else:
                #print(pl)
                kwargs = {'Fluid': Fluid, 'T0': T0, 'V': V}
                kwargs.update(v[1])
                all_volumes[v[0]] = kwargs
        #all_volumes += pl

    unique_volumes = list(set(list(all_volumes.keys())))

    #print('vols: ',all_volumes)

    for v in unique_volumes:

            host.add(v, Volume, **all_volumes[v])

    for pl in pipe_lines:

        c_item_last = None
        for i, c in enumerate(pl):
            if isinstance(c, str):
                c_item = host.get_item_name(c)
            else:
                c_item = host.get_item_name(c[0])

            if i > 0:
                host.add('Pipe_' + c_item_last.name + '_' + c_item.name, Pipe, kV=kV,
                         side1=c_item_last, side2=c_item)
            c_item_last = c_item

class EnergyMeter(B.Link):
    def construct_Link(self, **c):
        self.inlet = c['inlet']
        self.outlet = c['outlet']
        self.pipe = c['pipe']

        self.calc_funcs = [self.em_diff_f]

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'calc', 'val': 0},
            {'name': 'E', 'unit': 'kW', 'type': 'state', 'val': 0},
            {'name': 'W_MWh', 'unit': 'MWh/W', 'type': 'param', 'val': 1.0 / 1e6 / 3600},
            {'name': 'P_ewma', 'unit': 'kW', 'type': 'state', 'val': 0},

            #{'name': 'E', 'unit': 'MWh', 'type': 'state', 'val': 0},
        ]

    def em_diff_f(self):
        P=(outlet.T-inlet.T)*4187*pipe.m_dot/1000
        E_dot = P * W_MWh
        P_ewma_dot=(P-P_ewma)/1

        # print(kV)


class Valve_Fixed_Massflow(B.Link):
    def construct_Link(self, m_dot, select1=1, **c):
        self.inlet = c['inlet']
        self.out1 = c['out1']
        self.out2 = c['out2']

        self.calc_funcs = [self.Pipe_diff_f]

        self.Variables = [
            {'name': 'm_dot', 'unit': 'kg/s', 'type': 'param', 'val': m_dot},
            {'name': 'select1', 'unit': 'kg/s', 'type': 'param', 'val': select1},
        ]

    def Pipe_diff_f(self):
        T_pipe = inlet.T
        # H_dot=m_dot*((m_dot>0)*side1.H/side1.m+(m_dot<0)*side2.H/side2.m)
        H_dot = m_dot * ( (m_dot > 0) * inlet.h + (m_dot < 0) * ( out1.h * (select1>0) + out2.h * (select1<=0) ))
        inlet.m_dot = -m_dot
        out1.m_dot = m_dot*(select1>0)
        out2.m_dot = m_dot*(select1<=0)
        inlet.H_dot = -H_dot
        out1.H_dot = H_dot*(select1>0)
        out2.H_dot = H_dot*(select1<=0)


class Pipe_Fixed_Massflow(B.Link):
    def construct_Link(self, m_dot, **c):
        self.side1 = c['side1']
        self.side2 = c['side2']

        self.calc_funcs = [self.Pipe_diff_f]

        self.Variables = [
            {'name': 'm_dot', 'unit': 'kg/s', 'type': 'param', 'val': m_dot},
        ]

    def Pipe_diff_f(self):
        T_pipe = ((m_dot > 0) * side1.T + (m_dot <= 0) * side2.T)
        # H_dot=m_dot*((m_dot>0)*side1.H/side1.m+(m_dot<0)*side2.H/side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)
        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot
class Pipe_Liquid_Only(B.Link):
    def construct_Link(self,Fluid, kV, **c):
        self.side1=c['side1']
        self.side2=c['side2']

        self.Fluid=Fluid
        
        self.calc_funcs=[self.Pipe_diff_f]

        self.Variables = [

            {'name': 'kV', 'unit': 'kg/s/sqrt(kPa)', 'type': 'param', 'val': kV},
            {'name': 'm_dot_max', 'unit': 'kg/s', 'type': 'param', 'val': 100},
        ]

    def Pipe_diff_f(self):
        p = side1.p * (dp > 0) + side2.p * (dp <= 0)
        h = interp_bilin(Fluid.pq_h, ln(p), 0)

        T_pipe = ((m_dot > 0) * side1.T + (m_dot <= 0) * side2.T)
        dp = side1.p - side2.p
        m_dot_raw = sign(dp) * sqrt(abs(dp)) * kV
        m_dot = m_dot_raw * (abs(m_dot_raw) <= m_dot_max) + m_dot_max * (abs(m_dot_raw) > m_dot_max) * sign(
            m_dot_raw)

        H_dot = m_dot * h

        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot
                    
    def Pipe_diff_f2(self):
        
        
        dp=side1.p-side2.p
        p=side1.p*(dp>0)+side2.p*(dp<=0)
        h=interp_bilin(Fluid.pq_h,ln(p),0)
        
        m_dot=sign(dp)*sqrt(abs(dp))*kV    
        H_dot=m_dot*((m_dot>0)*h+(m_dot<0)*h)
        side1.m_dot=-m_dot
        side2.m_dot=m_dot
        side1.H_dot=-H_dot
        side2.H_dot=H_dot
        

class Pipe_Dim(B.Link):
    def construct_Link(self,Liquid, L=1, D=.05, **c):
        kV=3
        self.side1=c['side1']
        self.side2=c['side2']
        
        self.calc_funcs=[self.Pipe_diff_f_kV]
        
        self.liquid=Liquid
        
        self.Variables=[
                    {'name': 'kV', 'unit': 'kg/s/sqrt(kPa)', 'type':'param', 'val':kV},
                    {'name': 'mflow', 'unit': 'kg/s', 'type':'state', 'val':0},
                    {'name': 'kP', 'unit': 'm3/s/kPa', 'type':'param', 'val':5e-2},
                    {'name': 'rho', 'unit': 'kg/m3', 'type':'param', 'val':Liquid.rho},
                    {'name': 'pre_dp', 'unit': '1', 'type':'param', 'val':8/Liquid.rho*L/1000/math.pi**2/D**5},
                    {'name': 'pre_Re', 'unit': '1', 'type':'param', 'val':4/math.pi/D/1e-3}, 
                    {'name': 'T_pipe', 'unit': 'K', 'type':'param', 'val':20},
                    {'name': 'm_dot_max', 'unit': 'kg/s', 'type': 'param', 'val': 100},
                    ]

    def Pipe_diff_f(self):

        T_pipe=((m_dot>0)*side1.T+(m_dot<=0)*side2.T)
        dp=side1.p-side2.p

        Re=1200
        f=64/Re

        dp_calc=pre_dp*f*mflow**2

        mflow_dot=(abs(dp)-dp_calc)*kP
        # 
        m_dot=sign(dp)*mflow

     
        #H_dot=m_dot*((m_dot>0)*side1.H/side1.m+(m_dot<0)*side2.H/side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)
        side1.m_dot=-m_dot
        side2.m_dot=m_dot
        side1.H_dot=-H_dot
        side2.H_dot=H_dot
        #mflow=m_dot

    def Pipe_diff_f_kV(self):
        T_pipe = ((m_dot > 0) * side1.T + (m_dot <= 0) * side2.T)
        dp = side1.p - side2.p
        m_dot_raw = sign(dp) * sqrt(abs(dp)) * kV
        m_dot = m_dot_raw * (abs(m_dot_raw) <= m_dot_max) + m_dot_max * (abs(m_dot_raw) > m_dot_max) * sign(m_dot_raw)

        #H_dot = m_dot * ((m_dot > 0) * side1.H / side1.m + (m_dot < 0) * side2.H / side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)

        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot
        
class Var_Nozzle(B.Link):
    def construct_Link(self, m_dot=0, **c):
        self.side1=c['side1']
        self.side2=c['side2']

        self.calc_funcs=[self.Var_Nozzle_diff_f]

        self.Variables=[
                    {'name': 'flow', 'unit': 'kg/s', 'type':'param', 'val':m_dot}
                    ]
        
    def Var_Nozzle_diff_f(self):
        flow=flow
        #H_dot=flow*((flow>0)*side1.H/side1.m)
        H_dot = flow * ((flow > 0) * side1.h)
        side1.m_dot=-flow
        side2.m_dot=flow
        
        side1.H_dot=-H_dot
        side2.H_dot=H_dot
        
class Dual_Compressor(B.Link):
    def construct_Link(self,Fluid,N_comp,State1=0,State2=0,dT_Super=6, **c):
        self.Fluid=Fluid
        
        if 'dT_Super' in c:
            dT_Super=c['dT_Super']

        #dT_Super=10
       
        self.inlet=c['inlet']
        self.outlet=c['outlet']
        self.nozzle=c['nozzle']
        
        self.calc_funcs=[self.Compressor_diff_f]

        self.Variables=[
                    {'name': 'state', 'unit': '', 'type':'state', 'val':0},
                    {'name': 'dT_Super', 'unit': 'K', 'type':'param', 'val':dT_Super},
                    {'name': 'State1', 'unit': '', 'type':'param', 'val':State1},
                    {'name': 'State2', 'unit': '', 'type':'param', 'val':State2},
                    {'name': 'P_ewma', 'unit': '', 'type':'state', 'val':1},
                    {'name': 'Energy', 'unit': 'MJ', 'type':'state', 'val':0},
                    {'name': 'N_comp', 'unit': '1', 'type': 'param', 'val': N_comp},
                    ]
        
        self.m_dot_poly=np.array(c['polynom_m_dot'],dtype=np.float32)
        self.power_poly=np.array(c['polynom_power'],dtype=np.float32)
        
    def Compressor_diff_f(self):
        inlet_h=inlet.H/inlet.m
        
        T_Evap=interp_bilin(Fluid.pq_T,ln(inlet.p),0)
        T_Cond=interp_bilin(Fluid.pq_T,ln(outlet.p),1)

        dT_Super_Act=inlet.T-T_Evap
        print(dT_Super_Act)
        dT_Super_Err=dT_Super_Act-dT_Super
        
        S_in=interp_bilin(Fluid.ph_s,ln(inlet.p),inlet_h)
        rho_in=interp_bilin(Fluid.ph_rho,ln(inlet.p),inlet_h)
        #
        State1_in=(State1>0)*State1*(State1<=1)+(State1>1)
        State2_in = (State2 > 0) * State2 * (State2 <= 1) + (State2 > 1)
        state_set=(State1_in+State2_in)*(State1_in+State2_in<=N_comp)+(State1_in+State2_in>N_comp)*N_comp
        #state_set=State_set*(T_Evap>-10)
        state_dot=(state_set-state)*0.1
        # 
        p_in=inlet.p
        # 
        m_dot=calc_poly2x3(m_dot_poly,T_Evap,T_Cond)*state/3600
        m_dot=m_dot*(m_dot>=0)
        # 
        nozzle.flow=m_dot*(1.0+dT_Super_Err*.05)
        # 
        h_out_ideal=interp_bilin(Fluid.ps_h,ln(outlet.p),S_in)

        Power=calc_poly2x3(power_poly,T_Evap,T_Cond)*state
        Power = Power * (Power >= 0)
        P=Power*0.93
        P_ewma_dot=0.1*(Power/1000-P_ewma)
        Energy_dot = Power/1e6
        #
        inlet.m_dot=-m_dot
        outlet.m_dot=m_dot
        
        outlet_h=inlet_h+P/(m_dot+(m_dot==0))
        outlet.H_dot=outlet_h*m_dot
        inlet.H_dot=-inlet_h*m_dot

        #eff = (h_out_ideal-inlet_h)/(outlet_h-inlet_h)

        T_Discharge=interp_bilin(Fluid.ph_T,ln(outlet.p),outlet_h)
        print(T_Discharge)
        print(m_dot)
        # 
        # 
        p_out=outlet.p


class Transcritical_Compressor(B.Link):
    def construct_Link(self, Fluid, N_comp=1, State1=0, dT_Super=6, kP=0.05, kI=0, evap=None, **c):
        self.Fluid = Fluid
        self.evap = evap
        if 'dT_Super' in c:
            dT_Super = c['dT_Super']

        # dT_Super=10

        self.inlet = c['inlet']
        self.outlet = c['outlet']
        self.nozzle = c['nozzle']

        self.calc_funcs = [self.Compressor_diff_f]

        self.Variables = [
            {'name': 'state', 'unit': '', 'type': 'state', 'val': 0},
            {'name': 'dT_Super', 'unit': 'K', 'type': 'param', 'val': dT_Super},
            {'name': 'dT_Super_err_int', 'unit': 'K', 'type': 'state', 'val': 0},
            {'name': 'State1', 'unit': '', 'type': 'param', 'val': State1},
            #{'name': 'State2', 'unit': '', 'type': 'param', 'val': State2},
            {'name': 'P_ewma', 'unit': '', 'type': 'state', 'val': 1},
            {'name': 'Energy', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'N_comp', 'unit': '1', 'type': 'param', 'val': N_comp},
            {'name': 'kP', 'unit': '1', 'type': 'param', 'val': kP},
            {'name': 'kI', 'unit': '1', 'type': 'param', 'val': kI},

        ]

        self.m_dot_poly = np.array(c['polynom_m_dot'], dtype=np.float32)
        self.power_poly = np.array(c['polynom_power'], dtype=np.float32)

    def Compressor_diff_f(self):
        inlet_h = inlet.H / inlet.m

        T_Evap = interp_bilin(Fluid.pq_T, ln(evap.p), 0)
        T_Cond = interp_bilin(Fluid.pq_T, ln(outlet.p), 1)

        dT_Super_Act = inlet.T - T_Evap
        print(dT_Super_Act)
        dT_Super_Err = dT_Super_Act - dT_Super
        dT_Super_reg = dT_Super_err_int*kI
        dT_Super_err_int_dot = dT_Super_Err*((dT_Super_reg > -0.1)*(dT_Super_Err<0) + (dT_Super_reg< 0.1)*(dT_Super_Err>0))

        S_in = interp_bilin(Fluid.ph_s, ln(inlet.p), inlet_h)
        rho_in = interp_bilin(Fluid.ph_rho, ln(inlet.p), inlet_h)
        #
        State1_in = State1
            #(State1 > 0) * State1 * (State1 <= N_comp) + (State1 > N_comp)*N_comp

        state_set = State1_in * (State1_in <= 1.4)*(State1_in >= 0.5) + (State1_in > 1.4)

        state_dot = (state_set - state) * 0.1
        #
        p_in = inlet.p
        p_out = outlet.p

        #
        m_dot = calc_poly2x3(m_dot_poly, T_Evap, p_out/100) * state / 3600
        m_dot = m_dot * (m_dot >= 0)
        #
        nozzle.flow = m_dot * (1.0 + dT_Super_Err * kP + dT_Super_reg)
        #
        h_out_ideal = interp_bilin(Fluid.ps_h, ln(outlet.p), S_in)

        Power = calc_poly2x3(power_poly, T_Evap,  p_out/100) * state
        Power = Power * (Power >= 0)
        P = Power * 0.93
        P_ewma_dot = 0.1 * (Power / 1000 - P_ewma)
        Energy_dot = Power / 1e6
        #
        inlet.m_dot = -m_dot
        outlet.m_dot = m_dot

        outlet_h = inlet_h + P / (m_dot + (m_dot == 0))
        outlet.H_dot = outlet_h * m_dot
        inlet.H_dot = -inlet_h * m_dot

        # eff = (h_out_ideal-inlet_h)/(outlet_h-inlet_h)

        T_Discharge = interp_bilin(Fluid.ph_T, ln(outlet.p), outlet_h)
        print(T_Discharge)
        print(m_dot)
        #
        #


class Mass_Flow_Source(B.Link):
    def construct_Link(self,Fluid,m_dot, T, q, **c):
        self.Fluid=Fluid
        self.outlet=c['outlet']
        #self.parameters={self.gp('m_dot'): m_dot, self.gp('T'): T}
        
        
        if isinstance(Fluid,Fluid_Props):
                
            self.calc_funcs=[self.L2F.Mass_Flow_Source_diff_f]
            
            #[h]=self.Fluid.calc_props('Tq',[T,q],'h')
            h=ME.bilinear_interpolation(0, T, q, [0], self.Fluid.Tq_h)
            
            self.Variables=[
                        {'name': 'm_dot', 'unit': 'kg/s', 'type':'param', 'val':m_dot},
                        {'name': 'h', 'unit': 'J/kg', 'type':'param', 'val':h}
                        ]
        
        elif isinstance(Fluid,Liquid_Props):
        
            
            self.calc_funcs=[self.Liquid_Mass_Flow_Source_diff_f]
            
            self.Variables=[
                        {'name': 'm_dot', 'unit': 'kg/s', 'type':'param', 'val':m_dot},
                        {'name': 'H_dot', 'unit': 'J/kg', 'type':'param', 'val':Fluid.Cp*(T+273.15)*m_dot}
                        ]
        else:
            print(Fluid)
            raise ValueError('The medium class not recognized!')
        
    def F2P_Mass_Flow_Source_diff_f(self):
        
        
        outlet.m_dot=m_dot
        outlet.H_dot=m_dot*h       
        
    def Liquid_Mass_Flow_Source_diff_f(self):
        
        
        outlet.m_dot=m_dot
        outlet.H_dot=H_dot
        
class Pump_Fixed_Displacement_Reservoir(B.Link):
    def construct_Link(self,Liquid, V_disp=.5/3600/50,RPM=50, T=20, **c):
        if 'V_disp' in c:
            V_disp=c['V_disp']
            
        if 'RPM' in c:
            RPM=c['RPM']
        self.Liquid=Liquid     
        self.outlet=c['outlet']
        
        self.calc_funcs=[self.Pump_Fixed_Displacement_Reservoir_diff_f]

        self.Variables=[
                    {'name': 'V_disp', 'unit': 'm3/s', 'type':'param', 'val':V_disp},
                    {'name': 'RPM', 'unit': 'min^-1', 'type':'param', 'val':RPM},
                    {'name': 'Cp', 'unit': 'J/K', 'type':'param', 'val':self.Liquid.Cp},
                    {'name': 'T', 'unit': 'K', 'type':'param', 'val':T}
                    ]
        
    def Pump_Fixed_Displacement_Reservoir_diff_f(self):
        
        RPM=RPM*(RPM<=50)+(RPM>50)*50
        m_dot=V_disp*RPM*1000*(RPM>0)
        outlet.m_dot=m_dot
        outlet.H_dot=m_dot*(T+273.15)*Cp
        
#         
#         
class ThreeWayValve(B.Link):
    def construct_Link(self,kV, AC=0,**c):
        self.A=c['A']
        self.B=c['B']
        self.C=c['C']
        
        self.Variables=[
                    {'name': 'kV', 'unit': '', 'type':'param', 'val':kV},
                    {'name': 'AC', 'unit': 'min^-1', 'type':'param', 'val':AC}
                    ]
        self.calc_funcs=[self.ThreeWay_diff_f]
        
    def ThreeWay_diff_f(self):
        
        dp=A.p*(AC>0)+B.p*(AC<=0)-C.p
        m_dot=sign(dp)*sqrt(abs(dp))*kV    
        #H_dot=m_dot*((m_dot>0)*(A.H/A.m*(AC>0.1)+B.H/B.m*(AC<=0.1))+(m_dot<0)*C.H/C.m)
        H_dot = m_dot * ((m_dot > 0) * (A.h * (AC > 0.1) + B.h * (AC <= 0.1)) + (m_dot < 0) * C.h)
        
        A.m_dot=-m_dot*(AC>0.1)
        A.H_dot=-H_dot*(AC>0.1)
        B.m_dot=-m_dot*(AC<=0.1)
        B.H_dot=-H_dot*(AC<=0.1)
        C.m_dot=m_dot
        C.H_dot=H_dot

class Simple_Valve_Regulator(B.Node):
    def construct_Node(self, kP=0.1, kV=1, ini=0,**c):
        self.Variables=[
                    {'name': 'S_act', 'unit': '1', 'type':'state', 'val':ini},
                    {'name': 'S_set', 'unit': '1', 'type':'param', 'val':ini},
                    {'name': 'kP', 'unit': 'Hz', 'type':'param', 'val':kP},
                    {'name': 'kV', 'unit': '', 'type':'param', 'val':kV},
                    ]
                    
        self.input1=c['input1']
        self.input2=c['input2']
        
        self.calc_funcs=[self.Diff_f]
        
    def Diff_f(self):
        S_set_in = (S_set > 0) * S_set * (S_set <= 1) + (S_set > 1)
        S_act_dot=(S_set_in-S_act)*kP


        
        input1.kV=kV*S_act
        input2.kV=kV*(1-S_act)
        
        
                    
class ThreeWayValve_Regulated(B.Subsystem):
    def construct_Subsystem(self, fluid, kV, kP, T_ini=20, AC=0, V=0.1, pipe_types=[(Pipe, {}), (Pipe, {}), (Pipe, {})], **c):
        
        self.Vol=self.add('Vol',Volume,fluid,V,0,T_ini)
        
        pipeA=self.add('pipe_A',pipe_types[0][0],kV,
            side1=c['A'],
            side2=self.Vol,
            **pipe_types[0][1]
                       )
        
        pipeB=self.add('pipe_B',pipe_types[1][0],kV,
            side1=c['B'],
            side2=self.Vol,
                       **pipe_types[1][1]
                       )
        
        self.add('pipe_C',pipe_types[2][0],kV,
            side1=self.Vol,
            side2=c['C'],
                 **pipe_types[2][1])
            
        self.Reg=self.add('Reg',Simple_Valve_Regulator,kP,kV,AC,
                input1=pipeA, input2=pipeB)


class Valve_Regulated(B.Subsystem):
    def construct_Subsystem(self, kV=5, S_set_in=0, kP=0.1,**c):

        self.pipe = self.add('pipe', Pipe, kV*S_set_in,
                         side1=c['side1'],
                         side2=c['side2'])

        self.Variables = [
            {'name': 'S_set_in', 'unit': '', 'type': 'param', 'val': S_set_in},
            {'name': 'S_set', 'unit': '', 'type': 'param', 'val': S_set_in},
            {'name': 'S_act', 'unit': '', 'type': 'state', 'val': S_set_in},
            {'name': 'kV', 'unit': '', 'type': 'param', 'val': kV},
            {'name': 'kP', 'unit': '', 'type': 'param', 'val': kP}

        ]
        self.calc_funcs = [self.valve_calc]

    def valve_calc(self):
        S_set_in = (S_set > 0) * S_set * (S_set <= 1) + (S_set > 1)
        S_act_dot = (S_set_in - S_act) * kP

        pipe.kV = kV * S_act

        
class Pipe_Check(B.Link):
    def construct_Link(self,kV, reverse=False, **c):
        if reverse:
            self.outlet=c['inlet'] if 'inlet' in c else c['side1']
            self.inlet=c['outlet'] if 'outlet' in c else c['side2']
        else:
            self.inlet = c['inlet'] if 'inlet' in c else c['side1']
            self.outlet = c['outlet'] if 'outlet' in c else c['side2']

            #self.parameters={self.gp('kV'): kV}
    
        self.Variables=[
                    {'name': 'kV', 'unit': '', 'type':'param', 'val':kV}
                    ]
        self.calc_funcs=[self.Pipe_Check_diff_f]

        
    def Pipe_Check_diff_f(self):
        T_pipe = inlet.T
        dp=(inlet.p-outlet.p)#-10)
        ip=inlet.p
        print(ip)
        op=outlet.p
        print(op)
        m_dot=sqrt(abs(dp))*kV*(dp>0)

        #H_dot=m_dot*inlet.H/inlet.m
        H_dot = m_dot * inlet.h
        print(m_dot)
        inlet.m_dot=-m_dot
        inlet.H_dot=-H_dot
        
        outlet.m_dot=m_dot
        outlet.H_dot=H_dot

class Pipe_w_Power(B.Subsystem):
    def construct_Subsystem(self, P=0, m_dot=10, side1=None, side2=None, **c):
        self.side1 = side1
        self.side2 = side2


        self.Variables = [
            {'name': 'P', 'unit': '', 'type': 'param', 'val': P},
            {'name': 'm_dot', 'unit': '', 'type': 'param', 'val': m_dot}
            #{'name': 'P', 'unit': '', 'type': 'param', 'val': P}
        ]
        self.calc_funcs = [self.power]

    def power(self):


        #H_dot = m_dot * ((m_dot > 0) * side1.H / side1.m + (m_dot < 0) * side2.H / side2.m)
        H_dot = m_dot * ((m_dot > 0) * side1.h + (m_dot < 0) * side2.h)
        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot+P