from sim_components.generic.items import Subsystem, Node, SubsystemPrescribed
from sim_components.regulators.Static_Regulator import Static_Regulator
from sim_components.Building.climate_file_import import global_indata_reader
import numpy as np
import pvlib
import pandas as pd

f_m =1000
def add_schedule(item, schedule, schedule_key=None, schedule_key_static=None):
    print(item.get_path(schedule_key))
    if not type(schedule) == type(None) and schedule_key and schedule_key_static:

        static_data = {
            'data': {
                't': schedule['t'],
                'tags': {
                    item.get_path(schedule_key): schedule[schedule_key_static]
                }
            }
        }
        item.add('static', Static_Regulator, static_data)

class Air_h(Node):
    def construct_Node(self, T, humid):
        self.add_var('p_air', val=1.013e5, bind=True)
        self.add_var('c_pa', val=1.006*f_m, bind=True)
        self.add_var('Rg', val=0.62198, bind=True)
        self.add_var('c_pw', val=1.84*f_m, bind=True)
        self.add_var('h_we', val=2501*f_m, bind=True)
        self.add_var('humid', val=humid, bind=True)
        self.add_var('T', val=T, bind=True)
        self.add_var('h', val=76, bind=True)
        self.add_var('rho', val=1, bind=True)

        self.calc_funcs.append(self.calc)

    def calc(self):
        p_s = exp(12.03 - 4025 / (T + 235)) * 1e5
        p_v = humid * p_s

        x = Rg * p_s * humid / (p_air - p_v)
        h = c_pa * T + x * (h_we + c_pw * T)

        rho = (p_v * 0.018 + (p_air - p_v) * 0.029) / (8.314 * (T + 273.15))

class Air(Node):
    def construct_Node(self, T, humid, rho0, x0):


        self.add_var('p_air', val=1.013e5, bind=True)
        self.add_var('c_pa', val=1.006*f_m, bind=True)
        self.add_var('Rg', val=0.62198, bind=True)
        self.add_var('c_pw', val=1.84*f_m, bind=True)
        self.add_var('h_we', val=2501*f_m, bind=True)
        self.add_var('humid', val=humid, bind=True)
        self.add_var('T', val=T, bind=True)

        self.add_var('rho', val=rho0, bind=True)
        self.add_var('x', val=x0, bind=True)

        p_s = np.exp(12.03 - 4025 / (T + 235)) * 1e5
        p_v = humid * p_s

        x = self.Rg * p_s * humid / (1e5 - p_v)
        h = self.c_pa * T + self.x * (self.h_we + self.c_pw * T)

        self.add_var('h', val=h, bind=True)

        self.calc_funcs.append(self.calc)

    def calc(self):


        #h = c_pa * T + x * (h_we + c_pw * T)
        T = (h - x * h_we)/(c_pa + x * c_pw)

        p_s = exp(12.03 - 4025 / (T + 235)) * 1e5


        # x = Rg * p_s * humid / (p_air - p_v)
        humid = x * (p_air - humid * p_s) / (Rg * p_s)

        #humid + humid * p_s  / (Rg * p_s) = x * (p_air)/ (Rg * p_s)
        #humid (1 + 1/Rg) = x * (p_air)/ (Rg * p_s)
        humid = x * p_air / (p_s * (Rg + 1))
        p_v = humid * p_s

        rho = (p_v * 0.018 + (p_air - p_v) * 0.029) / (8.314 * (T + 273.15))

class Ambient(Subsystem):
    def construct_Subsystem(self, climate_file, climate_file_type, len_schedule, demand_register_callback=None, schedule=None):

        self.air = self.add('air', Air_h, 20, 50)

        self.climate_df, self.climate_meta_data = pvlib.iotools.read_epw(climate_file)

        df_solar = pvlib.solarposition.get_solarposition(
            self.climate_df.index,
            self.climate_meta_data['latitude'],
            self.climate_meta_data['longitude'],
            altitude=self.climate_meta_data['altitude'],
            temperature=self.climate_df['temp_air'],
            pressure=self.climate_df['atmospheric_pressure']
        )

        self.climate_df = pd.concat([self.climate_df, df_solar], axis=1)

        data_len = len(self.climate_df)


        self.t = [i*3600 for i in range(len_schedule)]

        static_data = {
            'data': {
                't': self.t,
                'tags': {
                    self.get_path('T'): [self.climate_df['temp_air'][i%data_len] for i in range(len_schedule)],
                    #self.get_path('I_direct_normal'): [df['Irradiance_Normal'][i%data_len] for i in range(len_schedule)],
                    #self.get_path('I_diffuse_horizontal'): [df['Irradiance_Horizontal'][i % data_len] for i in range(len_schedule)],
                    self.get_path('wind_speed'): [self.climate_df['wind_speed'][i % data_len] for i in range(len_schedule)],
                    self.get_path('relative_humidity'): [self.climate_df['relative_humidity'][i % data_len] for i in range(len_schedule)],
                }
            }
        }
        self.add('data', Static_Regulator, static_data)

        self.add_var('T_ground', val=10)

        self.calc_funcs.append(self.calc)

    def calc(self):
        air.T = T
        air.humid = relative_humidity/100

class Zone(Subsystem):
    def construct_Subsystem(self, area, volume, T0, RH0, demand_register_callback=None, schedule=None):
        volume *=1


        p_air = 1e5
        p_s = np.exp(12.03 - 4025 / (T0 + 235)) * p_air
        p_v = RH0/100 * p_s
        rho_water =  p_v*0.018/(8.314*(T0+273.15))
        rho_air =  (p_air-p_v)*0.029/(8.314*(T0+273.15))

        self.air = self.add('air', Air, T0, RH0 / 100, rho_air+rho_water, rho_water/rho_air)


        self.add_var('area', val=area, bind=True)

        self.add_var('m_water', vartype='state', val=rho_water*volume, bind=True)
        self.add_var('m_air', vartype='state', val=rho_air*volume, bind=True)

        #print(self.air.h)

        self.add_var('H', vartype='state', val=self.air.h*(self.m_water + self.m_air))

        #print(self.air.h*(self.m_water + self.m_air))
        #print('tho: ', rho_water + rho_air)
        ##print('rho air: ', rho_air)
        #print('m air: ', self.m_water + self.m_air)
        self.calc_funcs.append(self.calc)


    def calc(self):
        m_total = m_water + m_air
        air.x = m_water/m_total
        air.h = H/m_total

def calc_total_irradiance(df, tilt, azimuth):
    irradiance_total = []
    for r in df.iterrows():
        r = r[1]

        irr = pvlib.irradiance.get_total_irradiance(tilt, azimuth, r['zenith'], r['azimuth'], r['dhi'], r['ghi'], r['dhi'],
                                                    dni_extra=r['etrn'])
        irradiance_total.append(irr['poa_global'])

    return irradiance_total

class Ext_Wall(Subsystem):
    def construct_Subsystem(self, zone, ambient, area, fraction_windows, U_wall, U_windows, G_windows, tilt=90, azimuth=0, demand_register_callback=None, schedule=None):
        self.zone = zone
        self.ambient = ambient

        self.add_var('h_wall', val=U_wall * area)
        self.add_var('h_windows', val=U_windows * fraction_windows*area)
        self.add_var('g_windows', val=G_windows * fraction_windows*area)

        tot_irr = calc_total_irradiance(self.ambient.climate_df, tilt, azimuth)
        add_schedule(self, {'t': ambient.t, 'gain': [tot_irr[i%len(tot_irr)] for i in range(len(ambient.t))]}, schedule_key='irr', schedule_key_static='gain')

        self.calc_funcs.append(self.calc)

    def calc(self):
        dT_ambient_zone = ambient.T - zone.air.T
        P_transmission_wall = h_wall * dT_ambient_zone
        P_transmission_windows = h_windows * dT_ambient_zone
        P_radiation_windows = g_windows * irr

        zone.H_dot = P_transmission_wall + P_transmission_windows + P_radiation_windows

class Ext_Slab(Subsystem):
    def construct_Subsystem(self, zone, ambient, area, U, c_slab, thickness, T0, h_zone, demand_register_callback=None, schedule=None):
        self.zone = zone
        self.ambient = ambient

        self.add_var('T', val=T0, vartype='state')
        self.add_var('h', val=area*U)
        self.add_var('area', val=area)
        self.add_var('h_zone', val=area * h_zone)
        self.add_var('C', val=c_slab*area*thickness)

        self.calc_funcs.append(self.calc)

    def calc(self):
        dT_ground_slab = ambient.T_ground - T
        dT_slab_zone = T - zone.air.T

        P_ground = dT_ground_slab * h
        P_zone = dT_slab_zone * h_zone

        zone.H_dot = P_zone
        T_dot = (P_ground - P_zone)/C

class Roof(Subsystem):
    def construct_Subsystem(self, zone, ambient, area, U, c_roof, thickness, fraction_windows, U_windows,
                            G_windows, T0, h_conv, h_zone, tilt=0, azimuth=0, demand_register_callback=None, schedule=None):
        self.zone = zone
        self.ambient = ambient

        area_roof = (1-fraction_windows)*area

        self.add_var('T', val=T0, vartype='state')
        self.add_var('h', val=area_roof*U)
        self.add_var('h_conv', val=area_roof * h_conv)
        self.add_var('area', val=area_roof)

        # Radiation
        self.add_var('boltzmann', val=5.67e-8)  # 5.67 x 10-8W/m2K4
        self.add_var('emissivity', val=0.93)
        self.add_var('Taz', val=273.15)

        self.add_var('h_zone', val=area_roof * h_zone)
        self.add_var('C', val=c_roof*area*thickness)
        self.add_var('h_windows', val=U_windows * fraction_windows*area)
        self.add_var('g_windows', val=G_windows * fraction_windows*area)

        tot_irr = calc_total_irradiance(self.ambient.climate_df, tilt, azimuth)
        add_schedule(self, {'t': ambient.t, 'gain': [tot_irr[i%len(tot_irr)] for i in range(len(ambient.t))]}, schedule_key='irr', schedule_key_static='gain')

        self.calc_funcs.append(self.calc)

    def calc(self):
        dT_roof_zone = T - zone.air.T
        dT_ambient_zone = ambient.T - zone.air.T
        dT_ambient_roof = ambient.T - T
        #TODO h_conv a function of windspeed
        P_transmission_zone = h * dT_roof_zone
        P_transmission_windows = h_windows * dT_ambient_zone
        P_radiation_windows = g_windows * irr

        P_convection = dT_ambient_roof * h_conv
        P_radiation = emissivity*boltzmann*area*((ambient.T+Taz)**4-(T+Taz)**4)

        #TODO investigate this 1000
        zone.H_dot = P_transmission_zone + P_transmission_windows + P_radiation_windows
        P_solar_influx = irr*area*emissivity
        T_dot = (-P_transmission_zone + P_radiation + P_solar_influx + P_convection)/C



class InternalGain(Subsystem):

    def construct_Subsystem(self, zone, P_nominal, schedule_key='', schedule=None, demand_register_callback=None):
        self.zone = zone

        self.add_var('P_nominal', val=P_nominal)
        #self.add_var('gain', val=gain)

        add_schedule(self, schedule, schedule_key='gain', schedule_key_static=schedule_key)

        self.calc_funcs.append(self.calc)



    def calc(self):
        P = gain * P_nominal
        zone.H_dot = P

class InternalGain_Occupants(Subsystem):

    def construct_Subsystem(self, zone, P_occupant, N_occupants=1, schedule_key=None, schedule=None, demand_register_callback=None):
        self.zone = zone

        self.add_var('P_occupant', val=P_occupant)
        self.add_var('N_occupants', val=N_occupants)
        #self.add_var('N_occupants', val=N_occupants)

        add_schedule(self, schedule, schedule_key='N_occupants', schedule_key_static=schedule_key)

        self.calc_funcs.append(self.calc)

    def calc(self):
        P = P_occupant * N_occupants
        zone.H_dot = P

class DHW(Subsystem):

    def construct_Subsystem(self, ambient, F_nominal=1, T_surroundings=20, T_hot_water=55, schedule_key=None, schedule=None, demand_register_callback=None):


        self.add_var('F_nominal', val=F_nominal)
        self.add_var('gain', val=1)
        self.add_var('T_cold_water', val=5)
        self.add_var('T_hot_water', val=55)
        self.add_var('P_VVC', val= 0.1 * (T_hot_water - T_surroundings))

        def rotate(l, n):
            return l[-n:] + l[:-n]

        T_Cold_filtered = 5
        T_Cold = []
        for T_a in ambient.climate_df['temp_air']:
            T_Cold_filtered += (T_a - T_Cold_filtered) * 0.001
            T_Cold.append(T_Cold_filtered)
        T_Cold = rotate(list(np.clip(T_Cold, 2, 20)), 0)
        T_Cold_ext = [T_Cold[i%len(T_Cold)] for i in range(len(ambient.t))]
        #print([T_Cold[i%len(T_Cold)] for i in range(len(ambient.t))])
        add_schedule(self, {'t': ambient.t, 'gain': T_Cold_ext},
                     schedule_key='T_cold_water', schedule_key_static='gain')

        add_schedule(self, schedule, schedule_key='gain', schedule_key_static=schedule_key)
        if demand_register_callback:
            demand_register_callback('domestic_hot_water', self.get_path('P_VV'))
            demand_register_callback('domestic_hot_water_recirculation', self.get_path('P_VVC'))
            demand_register_callback('domestic_hot_water_temperature', self.get_path('T_hot_water'))
            #demand_register_callback('domestic_cold_water_temperature', self.get_path('T_cold_water'))

        self.calc_funcs.append(self.calc)

    def calc(self):
        F = F_nominal * gain
        P_VV = F / 4182 / (T_hot_water - 1)


class InternalCapacitance(Subsystem):

    def construct_Subsystem(self, zone, C, h, T0, demand_register_callback=None, schedule=None):
        self.zone = zone

        self.add_var('P', val=0)
        self.add_var('T', val=T0, vartype='state')
        self.add_var('C', val=C)
        self.add_var('h', val=h)

        self.calc_funcs.append(self.calc)

    def calc(self):
        dT_zone = T - zone.air.T
        P = h * dT_zone
        zone.H_dot = P
        T_dot = -P/C

class InfiltrationLoss(Subsystem):

    def construct_Subsystem(self, zone, F, demand_register_callback=None, schedule=None):
        self.zone = zone

        self.add_var('F', val=F)

        self.calc_funcs.append(self.calc)

    def calc(self):
        zone.H_dot = zone.air.h * F


class AHU(Subsystem):

    def construct_Subsystem(self, zone, ambient, F_nominal, F_infiltration_50Pa, T_set_zone_cooling,
                            T_set_zone_heating, k_supply,S_set_min=0, k_set_fan=1, eff=0.8, schedule_key=None,
                            schedule=None, demand_register_callback=None):
        self.zone = zone
        self.ambient = ambient

        #self.ambient_air = self.add('ambient_air', Air, 20, 50)
        self.supply_air = self.add('supply_air', Air_h, T_set_zone_heating, 50/100)

        self.add_var('F_nominal', val=F_nominal/1000*zone.area)
        self.add_var('F_infiltration', val=F_infiltration_50Pa*zone.area*0.00127/10*1e5/50/1000)

        self.add_var('eff', val=eff)
        self.add_var('T_set_zone_cooling', val=T_set_zone_cooling)
        self.add_var('T_set_zone_heating', val=T_set_zone_heating)

        self.add_var('Heating', val=0)
        self.add_var('Cooling', val=1)
        self.add_var('k_set_fan', val=k_set_fan)
        self.add_var('S_set_min', val=S_set_min)


        self.add_var('k_supply', val=k_supply)

        if demand_register_callback:
            demand_register_callback('space_heating', self.get_path('P_supply_AHU_delivered'))


        add_schedule(self, schedule, schedule_key='gain', schedule_key_static=schedule_key)

        self.calc_funcs.append(self.calc)

    def calc(self):
        #ambient.air = ambient.T


        Cooling = zone.air.T > T_set_zone_cooling
        Heating = zone.air.T < T_set_zone_heating
        Deadband = (Cooling + Heating <= 0)

        T_set_zone = T_set_zone_cooling if Cooling else T_set_zone_heating

        dT_zone_setpoint = T_set_zone - zone.air.T

        dT_zone_supply_ = dT_zone_setpoint * k_supply
        dT_zone_supply = positive(min2(10, dT_zone_supply_)) if Heating else negative(max2(-10, dT_zone_supply_))

        supply_air.T = dT_zone_supply + T_set_zone

        F_exchange = min2(1, (positive(abs(dT_zone_supply_) - 10) * k_set_fan + S_set_min)) * F_nominal

        F_extract =  F_exchange* gain
        F_circulate = F_exchange * (1 - gain)
        F_supply = F_infiltration + F_extract + F_circulate

        #H_dot_extract = zone.air.h * F_extract
        #H_dot_supply = ambient.air.h * F_supply
        #H_dot_extract - H_dot_supply
        dH_dot_extract_supply = (zone.air.h - ambient.air.h) * F_extract * zone.air.rho

        P_HX_available = dH_dot_extract_supply * eff

        #
        P_supply_demand = F_supply * supply_air.h * supply_air.rho - ambient.air.h * F_extract * ambient.air.rho - zone.air.h * F_circulate * zone.air.rho

        P_HX_Useful =  positive(P_HX_available) if sign(P_supply_demand) > 0 else negative(P_HX_available)

        P_supply_AHU_demand =  max2(0, P_supply_demand - P_HX_Useful)*Heating + min(0,  P_supply_demand - P_HX_Useful)*Cooling

        P_supply_AHU_delivered = max2(0, P_supply_AHU_demand) * Heating + min(0, P_supply_AHU_demand)*Cooling

        P_AHU_undelivered = P_supply_AHU_demand - P_supply_AHU_delivered

       # P_HVAC =

        zone.H_dot = F_supply * (supply_air.h - zone.air.h)*ambient.air.rho - P_AHU_undelivered

        zone.m_water_dot = F_supply * (supply_air.x* supply_air.rho  - zone.air.x*zone.air.rho)
        zone.m_air_dot = F_supply * (zone.air.x*zone.air.rho - supply_air.x* supply_air.rho)

class Radiator(Subsystem):

    def construct_Subsystem(self, zone, T_set_zone_heating, k_supply, demand_register_callback=None, schedule=None):
        self.zone = zone

        self.add_var('T_set_zone_heating', val=T_set_zone_heating)



        self.add_var('k_supply', val=k_supply)

        if demand_register_callback:
            demand_register_callback('space_heating', self.get_path('P'))

        self.calc_funcs.append(self.calc)

    def calc(self):

        dT_zone_setpoint = T_set_zone_heating - zone.air.T

        P = positive(dT_zone_setpoint)*k_supply

        zone.H_dot = P


def make_interp_table(x, y):
    return np.concatenate([[len(x), min(x), max(x)], x, y])


class EnergyModel(Subsystem):
    def construct_Subsystem(self, **c):
        zones_specification = c['zones_specification']
        import json
        with open(c['schedule_file'], 'r') as f:
            schedule = json.load(f)

        for k, v in zones_specification['specification']['items'].items():
            v['demand_register_callback'] = self.register_demand
            v['schedule'] = schedule


        self.demand_links = {'space_heating': ('Stub.VS.P', [], 0),
                             'cooling': ('Stub.KB.P', [], 0),
                             'domestic_hot_water': ('Stub.VV.P', [], 0),
                             'domestic_hot_water_recirculation': ('Stub.VV.P_VVC', [], 0),
                             'domestic_hot_water_temperature': ('Stub.VV.T_set', [], 55),
                             'domestic_cold_water_temperature': ('Stub.EP.T_Cold', [], 10),
                             'waste_heat_regeneration': ('Stub.KB.P_process', [], 0)
                             }

        building = self.add('Building', SubsystemPrescribed, **zones_specification)

        self.generate_demand_calculations()

        # temperatures for demands
        self.T_VS_table = make_interp_table([-20., 20.], [80., 20.])
        self.T_VS_return_table = make_interp_table([-20., 20.], [60., 20.])

        self.T_KB_table = make_interp_table([-20., 20.], [7., 7.])
        self.T_KB_return_table = make_interp_table([-20., 20.], [12., 12.])

        self.calc_funcs.append(self.generate_demand_temps)

    def register_demand(self, demand_key, demand_var):
        print('Registering <', demand_var, '> as <', demand_key, '>')
        self.demand_links[demand_key][1].append(demand_var)

    def generate_demand_calculations(self):
        for k, v in self.demand_links.items():
            k_var = v[0]
            self.add_var(k_var, val=v[2])
            if len(v[1]) > 0:
                demand_str = k_var + ' = ' + '+'.join(v[1])
                print('Demand for ' + k + ': ', demand_str)
                self.add_calc_func_str(k_var, demand_str)

    def generate_demand_temps(self):

        Stub.VS.T_set = interp_lin(T_VS_table, Building.ambient.T)
        Stub.VS.T_return = interp_lin(T_VS_return_table, Building.ambient.T)

        Stub.KB.T_set = interp_lin(T_KB_table, Building.ambient.T)
        Stub.KB.T_return = interp_lin(T_KB_return_table, Building.ambient.T)

if __name__ == "__main__":
    from sim_tools.store.simulation_store_mongodb import Item_Factory
    import pandas as pd
    from datetime import datetime, timedelta
    #from  import *
    item_factory = Item_Factory('')

    H_free_full = 9.4
    area_06a = 8311
    T0 = 20

    first_year = 2020
    years = 2

    start = datetime(first_year,1,1,0)
    end = datetime(first_year+years,1,1,0)
    current = start
    datetime_list = []
    while current<end:
        datetime_list.append(current)
        current += timedelta(hours=1)


    schedule = pd.DataFrame({
        't': [i*3600 for i in range(len(datetime_list))],
        'datetime': datetime_list
    })

    schedule.set_index('datetime', inplace=True)




    schedule['zone06a_customers'] = 0

    monthly_distribution_visitation = [0.10709505, 0.07496653, 0.08032129, 0.09370817, 0.06157965,
                        0.05890228, 0.08032129, 0.07362784, 0.10843373, 0.08701473,
                        0.08701473, 0.08701473]
    Weekly_distribution_visitation = [0.81, 0.65, 0.75, 0.65, 0.92, 1.62, 1.62]
    daily_distribution_visitation = [0] * 10 + [0.6, 0.6, 0.6, 1.2, 1.2, 1.2, .9, 1.2, 1.2, 1.2, .93, 1.2] + [0] * 2

    zones_customer_densities = [1, 1, 0, 2.5, 1, 1, 0.5, 0, 1.5, 0.75, 1.5, 0, 0, 0]

    zone06a_customers_density = zones_customer_densities[6] / sum(zones_customer_densities)
    schedule['zone06a_customers'] = zone06a_customers_density*np.array([
        Weekly_distribution_visitation[d.weekday()] * daily_distribution_visitation[d.hour] * monthly_distribution_visitation[d.month - 1] / d.daysinmonth / 12 for d in schedule.index])

    #Staff

    zone06a_Weekly_distribution_staff = [[0]*9 + [1]*10 + [0]*4 for i in range(6)] + [[0]*24]

    zones_staff_densities = [0.5, 0.25, 1, 0.75, 0.25, 0.25, 0.25, 1, 0.5, 0.75, 1, 2, 0.25, 0.1]

    zone06a_staff_density = zones_staff_densities[6] / sum(zones_staff_densities)
    schedule['zone06a_staff'] = zone06a_staff_density * np.array([
        zone06a_Weekly_distribution_staff[d.weekday()][d.hour-1] for d in schedule.index])

    #AHU
    zone06a_Weekly_distribution_AHU = zone06a_Weekly_distribution_staff
    schedule['zone06a_AHU'] =  np.array([
        zone06a_Weekly_distribution_AHU[d.weekday()][d.hour - 1] for d in schedule.index])

    #Lighting

    zone06a_Weekly_distribution_lighting = [[0] * 6 + [0.5] * 4 + [1] * 10 + [0.25]*1 + [0]*3 for i in range(6)] + [[0] * 24]


    schedule['zone06a_lighting'] = np.array([
        zone06a_Weekly_distribution_lighting[d.weekday()][d.hour - 1] for d in schedule.index])

    # Equipment

    zone06a_Weekly_distribution_equipment = [[0] * 24 for i in range(7)]

    schedule['zone06a_equipment'] = np.array([
        zone06a_Weekly_distribution_equipment[d.weekday()][d.hour - 1] for d in schedule.index])



    default_building = {
        'ambient': {
            'item_class': 'sim_components.Building.building.Ambient',
            'climate_file': 'ARG_BARILOCHE-AERO_877650_IW2.epw',
            'climate_file_type': 'epw',
            'len_schedule': len(schedule)
        },
        'zone06a': {
            'item_class': 'sim_components.Building.building.Zone',
            'area': area_06a,
            'volume': H_free_full * area_06a,
            'T0': T0,
            'RH0': 50,
        },
        #'zone06a_ext_wall': {
        #    'item_class': 'sim_components.Building.building.Ext_Wall',
        #    'zone': '@zone06a',
        #    'ambient': '@ambient',
        #    'area': 13 * 8 * H_free_full,
        #    'fraction_windows': .35,
        #    'U_wall': .35,
        #    'U_windows': 2,
        #    'G_windows': 0.6
        #},
        #'zone06a_ext_slab': {
        #    'item_class': 'sim_components.Building.building.Ext_Slab',
        #    'zone': '@zone06a',
        #    'ambient': '@ambient',
        #    'area': area_06a,
        #    'U': 2.9,
        #    'c_slab': 1000,
        #    'thickness': 0.3,
        #    'T0': T0,
        #    'h_zone': 20
        #},
        #'zone06a_ext_roof': {
        #    'item_class': 'sim_components.Building.building.Roof',
        #    'zone': '@zone06a',
        #    'ambient': '@ambient',
        #    'area': area_06a,
        #    'U': .3,
        #    'c_roof': 1000,
        #    'thickness': 0.1,
        #    'fraction_windows': 203 / 21376,  # output excel 3 sim
        #    'U_windows': 2,
        #    'G_windows': 0.6,
        #    'T0': T0,
        #    'h_conv': 5,
        #    'h_zone': 5
        #},
        #'zone06a_equipment': {
        #    'item_class': 'sim_components.Building.building.InternalGain',
        #    'zone': '@zone06a',
        #    'P_nominal': 0,
        #    'schedule_key': 'zone06a_equipment',
        #    'schedule': schedule
        #},
        #'zone06a_lighting': {
        #    'item_class': 'sim_components.Building.building.InternalGain',
        #    'zone': '@zone06a',
        #    'P_nominal': 2.4 * area_06a,
        #    'schedule_key': 'zone06a_lighting',
        #    'schedule': schedule
        #},
        #'zone06a_customers': {
        #    'item_class': 'sim_components.Building.building.InternalGain_Occupants',
        #    'zone': '@zone06a',
         #   'P_occupant': 100,
         #   'schedule_key': 'zone06a_customers',
         #   'schedule': schedule

        #},
        #'zone06a_staff': {
        #    'item_class': 'sim_components.Building.building.InternalGain_Occupants',
        #    'zone': '@zone06a',
        #    'P_occupant': 100,
        #    'schedule_key': 'zone06a_staff',
        #    'schedule': schedule

        #},
        #'zone06a_furniture': {
        #    'item_class': 'sim_components.Building.building.InternalCapacitance',
        #    'zone': '@zone06a',
        #    'C': 200 * area_06a,
        #    'h': 20 * area_06a,
        #    'T0': T0
        #},
        #'zone06a_AHU': {
        #    'item_class': 'sim_components.Building.building.AHU',
        #    'zone': '@zone06a',
        #     'ambient': '@ambient',
        #     'F_nominal': 0.7, #L/s/m2
        #    'eff': 0.8,
        #    'schedule_key': 'zone06a_AHU',
        #    'schedule': schedule,
        #    'T_set_zone_heating': 20,
        #    'T_set_zone_cooling': 26,
        #    'F_infiltration_50Pa': 0.6, #0.6 L/s/m2
        #}

    }

    specification = {'specification': {'items': default_building}}

    Stub = item_factory.get_Item('Stub', '', SubsystemPrescribed, **specification)

    from sim_tools.assemble_solve.model_assemble import Model

    model = Model('', Stub)
    model.assemble('ALL')
    try:
        model.solve_t_out(schedule.t[:5], dt_fix=1, history=True)
    except:
        pass
    history = pd.DataFrame(model.history)

    plotkeys = ['Stub.zone06a.air.T']

    import matplotlib.pyplot as plt
    plt.show()


    for pk in plotkeys:
        history[pk].plot()

"""
equipment
'schedule': {
                'weekdays': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'friday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'saturday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'sunday': [0]*24,
            }
            
lighting
'schedule': {
                'weekdays': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'friday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'saturday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'sunday': [0]*24,
            }
            
customers
'schedule': {
                'weekdays': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'friday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'saturday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'sunday': [0]*24,
            }
            
staff
'schedule': {
                'weekdays': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'friday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'saturday': [0, 0, 0, 0,0,0,0.5,0.5,0.5,1,1,1,1,1,1,1,1,1,0.25,0,0,0],
                'sunday': [0]*24,
            }
            
class Building(Subsystem)

    def construct_Subsystem(self, zones):

        ambient = self.add('ambient', Ambient)

        # Self service, warehouse floor 0
        for zname, z in zones.items():
            z_ = self.add(zname, Zone, **z['zone'])

            self.add(zname+'_ext_wall', Ext_Wall, **z['ext_wall'])
            self.add(zname + '_ext_wall', Ext_Wall, **z['ext_wall'])
            self.add(zname + '_ext_slab', Ext_Wall, **z['extslab'])
            self.add(zname + '_ext_wall', Ext_Wall, **z['ext_wall'])
            self.add(zname + '_ext_wall', Ext_Wall, **z['ext_wall'])
        zone06a_ext_wall =
        zone06a_ext_slab = self.add('zone06a_ext_wall', Zone, **z06a['ext_wall'])
        zone02a_slab
        zone02a_roof
        zone02a_equipment
        zone02a_visitation
        zone02a_employees
        zone02a_internal_capacity
        zone02a_AHU


#---------Entrance floor 0----------
    zone02a = self.add('zone02a', Zone, **zones['zone02a'])
    zone02a_ext_wall = self.add('zone02a_ext_wall', Zone, **zones['zone02a'])
    zone02a_slab
    zone02a_equipment
    zone02a_visitation
    zone02a_employees
    zone02a_internal_capacity
    zone02a_AHU
    
#Showroom floor 1
zone_03a = self.add('zone_03a', Zone **zones['zone_03a'])
ceiling_03a = self.add('ceiling_03a', Roof, ambient=ambient, zone=zone_03a)

#Kitchen floor 1
zone04a
ceiling_04a = self.add('ceiling_04a', Roof, ambient=ambient, zone=zone_04a)
ext_wall_04a = self.add('ext_wall_04a', Ext_Wall, ambient=ambient, zone=zone_04a, area=z04a['ext_wall_area'], window_area=z04a['window_area'])
# Ikea restaurant and café floor 1
zone04b
ceiling_zone04b =

#Market hall floor 0

z05a = c['zone_05a']
slab_zone05a = self.add('slab_05a', Slab, ground=ground, zone=zone_05a, area=z05a['area'], h_zone=z05a['Area'], h_ground=z05a['h_ground'])

#Glass House floor 0
zone05b

z05b = c['zone_05b']
slab_zone05b = self.add('slab_05b', Slab, ground=ground, zone=zone_05b, area=z05b['area'], h_zone=z05b['Area'],
                        h_ground=z05b['h_ground'])


#Self service, warehouse floor 0
zone06a = self.add('zone06a', Zone, **zones['zone06a'])

#Goods receiving - 0 floor
zone08a

#Check out - cash area - floor 0
zone09a

#Customer service area - floor 0
zone10a

#Exit and Restaurant/Bistro - floor 0
zone11a

#Office - co-worker area floor 1
zone12a

#Mechanical technical floor 1
zone13a

#Emergency
zone13b



"""