from datetime import datetime, timedelta
import numpy as np

import pandas as pd

def read_iwec(file_in, t0=datetime(2016, 1, 1, 12)):
    # PRN IWEC file from http://www.equaonline.com/ice4user/indexIWEC.html
    # file_in = '106370_IWC.PRN'

    df_iwec = pd.read_csv(file_in, sep='\s+', decimal='.', engine='python')

    df_iwec = df_iwec.rename(columns={"#": "hours",
                                      'Time': 'T Ambient (C)',
                                      "TAir": "RelHumidity",
                                      "RelHum": "Windx",
                                      "WindX": "Windy",
                                      "WindY": "IDirNormal",
                                      "IDirNorm": "IDiffHorizontal",

                                      })
    df_iwec.pop('IDiffHor')

    df_iwec['date_time'] = [t0 + timedelta(seconds=t * 3600) for t in df_iwec['hours']]
    df_iwec.set_index('date_time', inplace=True)

    return df_iwec


def read_iwec2(file_in, t0=datetime(2016, 1, 1, 12), move_irrad=None):
    import pvlib
    df, meta_data = pvlib.iotools.read_epw(file_in, coerce_year=None)

    if move_irrad == 'horizontal':
        df['ghi'] += df['dni']
        df['dni'] = 0

    elif move_irrad == 'normal':
        df['dni'] += df['ghi']
        df['ghi'] = 0

    elif move_irrad == 'sum':
        df['dni'] += df['ghi']
        df['ghi'] = df['dni']

    df = df.rename(columns={
        # "hour": "hour",
        'temp_air': 'T Ambient (C)',
        "relative_humidity": "Humidity",
        "wind_speed": "Windspeed",
        #  "WindX": "Windy",
        "dni": "Irradiance_Normal",
        "ghi": "Irradiance_Horizontal",

    })
    # df['Irradiance_Normal']/=1000
    # df['Irradiance_Horizontal']/=1000
    df['hours'] = [i for i in range(len(df))]
    # df_iwec.pop('IDiffHor')
    df['date_time'] = [t0 + timedelta(seconds=t * 3600) for t in df['hours']]
    df.set_index('date_time', inplace=True)
    # df_iwec['date_time'] = [t0 + timedelta(seconds=t*3600) for t in df_iwec['hours']]
    # df_iwec.set_index('date_time',inplace=True)

    return df, meta_data


def convert_epw_json(file_in, file_out=None, t0=datetime(2016, 1, 1, 12)):
    import json
    df, meta = read_iwec2(file_in, t0=datetime(2016, 1, 1, 12))
    if not file_out:
        file_out = file_in.split('.')[0] + '.json'

    with open(file_out, 'w') as fo:
        json.dump({
            'df': df.to_json(),
            'meta': meta
        }, fo)


def read_muf(file_in, t0=datetime(2016, 1, 1, 0)):
    # PRN IWEC file from http://www.equaonline.com/ice4user/indexIWEC.html
    # file_in = 'Erlahour.dat'
    df_muf = pd.read_csv(file_in, sep='\s+', decimal='.', engine='python')

    # df['T Ambient (C)'] = df['Time']
    df_muf = df_muf.rename(columns={"hour": "hours",
                                    'T_ambient': 'T Ambient (C)',
                                    "A": "Humidity",
                                    "B": "Enthalpy",
                                    "C": "Windspeed",
                                    "D": "Irradiance_Normal",
                                    "E": "Irradiance_Horizontal",

                                    })
    # df_iwec.pop('IDiffHor')
    df_muf['hours_idx'] = df_muf['hours']
    df_muf.set_index('hours_idx', inplace=True)
    #
    # df_muf = remove_hours(df_muf, 0, 3)
    # df_muf=df_muf.dropna()
    df_muf['hours'] = [i for i in range(len(df_muf))]

    df_muf['date_time'] = [t0 + timedelta(seconds=t * 3600) for t in df_muf['hours']]
    df_muf.set_index('date_time', inplace=True)

    return df_muf


def read_indata_csv(file_in, t0=datetime(2016, 1, 1, 0)):
    # Read csv file
    df = pd.read_csv(file_in, sep=';', decimal=',')
    df['t'] = (df['hours'] - 1) * 3600
    df['date_time'] = [t0 + timedelta(seconds=t * 3600) for t in df['hours']]
    df.set_index('date_time', inplace=True)

    return df


def read_meteo_export(file_in):
    df = pd.read_csv(file_in, skiprows=11, sep=';')

    df['hours'] = [i + 1 for i, r in enumerate(df.iterrows())]
    df_out = pd.DataFrame({'hours': df['hours'].values, 'T Ambient (C)': df['Temperature  [2 m above gnd]'].values,
                           'Humidity': df['Relative Humidity  [2 m above gnd]'].values,
                           'wind_direction (deg)': df['Wind Direction  [10 m above gnd]'].values,
                           'Windspeed': df['Wind Speed  [10 m above gnd]'].values,
                           'Irradiance_Normal': df['Shortwave Radiation  [sfc]'].values,
                           'Irradiance_Horizontal': 0,

                           })

    # for r in df.iterrows():
    #    print(r)
    #    adsasd=asdsdf
    df_out['date_time'] = [
        datetime(int(r[1]['Year']), int(r[1]['Month']), int(r[1]['Day']), int(r[1]['Hour']), int(r[1]['Minute'])) for r
        in df.iterrows()]
    df_out.set_index('date_time', inplace=True)

    return df_out


def global_indata_reader(file_in, t0=datetime(2016, 1, 1, 0), key='epw'):
    if key == "epw":
        df, meta = read_iwec2(file_in, t0=t0)
        return df
    elif key == "epw_normal":
        df, meta = read_iwec2(file_in, t0=t0, move_irrad='normal')
        return df
    elif key == "epw_horizontal":
        df, meta = read_iwec2(file_in, t0=t0, move_irrad='horizontal')
        return df
    elif key == "epw_sum":
        df, meta = read_iwec2(file_in, t0=t0, move_irrad='sum')
        return df

    elif key == "muf":
        return read_muf(file_in, t0=t0)
    elif key == "csv":
        return read_indata_csv(file_in, t0=t0)
    elif key == "meteo_export":
        return read_meteo_export(file_in)
    else:
        raise ValueError('Unknown extension of climate file: <' + key + '>')

if __name__ == "__main__":
    df = global_indata_reader('ARG_BARILOCHE-AERO_877650_IW2.epw', key='epw')