import sim_components.generic.items as B
import numpy as np
class Prosumer(B.Node):
    def construct_Node(self,P=0):

        self.add_var('P', unit='kW', desc='The power produced by prosumer. Negative denotes consumption.')

    def output(self, sys_dict):
        #print('Power of ', self.gp(''), ' is: ', sys_dict[self.gp('P')], ' kW')
        pass
class WindGenerator(Prosumer):
    def construct_Node(self):

        super().construct_Node(P=0)

        self.add_var('v', unit='m/s', desc='Windspeed')
        self.add_var('v_mean', unit='m/s', val=10, desc='Windspeed')



        self.calc_funcs.append(self.calc)

    def calc(self):
        #v = rand(v_mean)
        P = sqrt(v)



    def output(self, sys_dict):
        super().output(sys_dict)
        sys_dict[self.gp('v')] = np.random.rayleigh(10)
        #print('Windspeed of ', self.gp(''), ' is: ', sys_dict[self.gp('v')], ' m/s')


class SolarPanel(Prosumer):
    def construct_Node(self):

        super().construct_Node(P=0)

        self.add_var('I', unit='W/m2', desc='Solar influx')
        self.add_var('A', unit='m2', desc='Area', val=1)
        self.add_var('efta', unit='1', desc='efficiency')

        self.calc_funcs.append(self.calc)

    def calc(self):

        P = I * A * efta


class Building(Prosumer):
    def construct_Node(self):

        super().construct_Node(P=0)

        self.add_var('t', unit='h', vartype='state', desc='time from start')
        self.add_var('h', unit='h/s', val=1./3600, desc='time from start')
        self.add_var('hd', unit='h/day', val=24, desc='time from start')

        self.add_var('tod', unit='h', desc='time of day')
        self.add_var('pi', unit='1', desc='pi', val=np.pi)
        self.add_var('P_max', val=5, unit='kW', desc='Maximum discharging rate')


        self.calc_funcs.append(self.diff)

    def diff(self):
        t_dot = h
        tod = t % hd

        P = -abs(sin(tod/hd*pi))*P_max

    def output(self, sys_dict):
        super().output(sys_dict)
        #print('Sime time in hours: ',sys_dict[self.gp('t')],' Hour of day: ',sys_dict[self.gp('tod')])


class Battery(Prosumer):
    def construct_Node(self, battery_capacity):
        super().construct_Node(P=0)

        self.add_var('E_cap', unit='kWh', val=battery_capacity, desc='Capacity')
        self.add_var('Rel_Charge', unit='%', vartype='state', desc='Charging state')
        self.add_var('P_set', val=0, unit='kW', desc='Discharging rate requested')
        self.add_var('P_max', val=5, unit='kW', desc='Maximum discharging rate')

        self.calc_funcs.append(self.diff)

    def diff(self):
        P = min(P_set, P_max) * (Rel_Charge > 0)*(P_set >0) - (P_set < 0)* (Rel_Charge < 100) * min(-P_set, P_max)
        Rel_Charge_dot = -P / 3600 / E_cap * 100

    def output(self, sys_dict):
        super().output(sys_dict)
        #print('P setpoint: ', sys_dict[self.gp('P_set')])
        #print('Charge: ', sys_dict[self.gp('Rel_Charge')],' %')


class Controller(B.Subsystem):
    def construct_Subsystem(self, prosumers, rel_charge_min=10):

        self.add_var('P_balance_wo_battery', val=0, unit='kW', desc='Balance of product/consumption in non-battery prosumers')
        self.add_var('P_balance', val=0, unit='kW', desc='Balance of product/consumption in all prosumers')
        self.add_var('Rel_Charge_Min', val=rel_charge_min, unit='%', desc='Min charge state')
        # Calculate the energy balance of the system and add/remove power as needed by the grid as an infinite ressource
        self.prosumers = prosumers

        non_battery_prosumers = []
        non_battery_prosumers_p_strs = []
        batteries = []
        for p in self.prosumers:
            if not isinstance(p, Battery):
               non_battery_prosumers.append(p)
               non_battery_prosumers_p_strs.append(p.gp('P'))
            else:
                batteries.append(p)

        assert len(batteries) == 1, 'There most be one and only one battery in the system, seems there is: ' + str(
            len(batteries)) + ' in this system!'

        self.battery = batteries[0]


        self.add_calc_func_str('P_balance_wo_battery','P_balance_wo_battery = ' + '+'.join(non_battery_prosumers_p_strs))

        self.add_calc_func_str('P_balance' ,'P_balance = ' + '+'.join([p.gp('P') for p in self.prosumers]))

        self.calc_funcs.append(self.diff)

    def diff(self):
        battery.P_set = - P_balance_wo_battery * ((battery.Rel_Charge > Rel_Charge_Min) * (P_balance_wo_battery < 0) + (P_balance_wo_battery >= 0))

    def output(self, sys_dict):
        pass
        #print('P balance wo Battery: ',sys_dict[self.gp('P_balance_wo_battery')])
        #print('P balance: ',sys_dict[self.gp('P_balance')])

    # def output(self, sys_dict):
    #     P_balance = 0
    #     batteries = []
    #
    #     for p in self.prosumers:
    #         if not isinstance(p, Battery):
    #             P_balance += sys_dict[p.gp('P')]
    #         else:
    #             batteries.append(p)
    #
    #     assert len(batteries) == 1, 'There most be one and only one batteri in the system, seems there is: ' + str(len(batteries)) + ' in this system!'
    #     sys_dict[batteries[0].gp('P_set')] = -P_balance


class HybridEnergySystem(B.Subsystem):
    # Calculate the energy balance of the system and add/remove power as needed by the grid as an infinite ressource

    def construct_Subsystem(self, battery_capacity=10):

        #Create prosumers
        self.prosumers = []

        self.prosumers.append(self.add('Wind', WindGenerator))
        self.prosumers.append(self.add('Solar', SolarPanel))
        self.prosumers.append(self.add('Building', Building))
        self.prosumers.append(self.add('Battery', Battery, battery_capacity))

        self.controller = self.add('Controller', Controller, self.prosumers)


class GenericSystem(B.Subsystem):
    # Calculate the energy balance of the system and add/remove power as needed by the grid as an infinite ressource

    def construct_Subsystem(self):
        pass

if __name__ == "__main__":
    from sim_tools.assemble_solve.model_assemble import Model
    import pandas as pd

    item_fact = B.Item_Factory('itemfact')
    hybrid_system = HybridEnergySystem('Stub','',item_fact)

    model = Model('hybrid_system', hybrid_system)
    model.assemble(['ALL'])
    for i in range(24*365):
        model.step_update(3600, dt_fix=60, history=True)

    df = pd.DataFrame(model.history)

    #print(df.describe())
    #print(list(df))

    df['Stub.Battery.Rel_Charge'].plot()
