import sim_components.generic.items as BC
from sim_components.network.cost_optim import generate_orders, generate_orders_sorted
from time import time

class ThermalChannel:
    def __init__(self,name,node_vol,T0,kV_m,Fluid):
        self.name=name
        self.node_vol=node_vol
        self.T0 = T0
        self.kV_m = kV_m
        self.Fluid = Fluid


class Transmission_Volume(BC.Node):
    def construct_Node(self, Fluid, V, q0=0, T0=20, p0=0):

        self.calc_funcs = [self.Liquid_Volume_diff_f]
        self.Fluid = Fluid
        self.Liquid = Fluid

        m0 = self.Fluid.rho * V * 1.1

        self.Variables = [
            {'name': 'm', 'unit': 'kg', 'type': 'state', 'val': m0},
            {'name': 'H', 'unit': 'J', 'type': 'state', 'val': self.Fluid.Cp * (T0 + 273.15) * m0},
            {'name': 'h', 'unit': 'J', 'type': 'param', 'val': self.Fluid.Cp * (T0 + 273.15)},
            {'name': 'q', 'unit': '1', 'type': 'param', 'val': 1},
            {'name': 'p', 'unit': 'kPa', 'type': 'param', 'val': 1e3},
            {'name': 'T', 'unit': 'K', 'type': 'param', 'val': T0},
            {'name': 'V', 'unit': 'm3', 'type': 'param', 'val': V},
            {'name': 'Cp', 'unit': 'J/K', 'type': 'param', 'val': self.Fluid.Cp},
            {'name': 'rho0', 'unit': 'kg/m3', 'type': 'param', 'val': self.Fluid.rho},
            {'name': 'kP', 'unit': 'Hz', 'type': 'param', 'val': .1}
        ]

    def Liquid_Volume_diff_f(self):
        T = H / Cp / m - 273.15
        h = H / m

        d_rho = (m / V - rho0) / rho0
        p = d_rho * 1e4 + 1e3

        #p_dot = (rp - p) * kP
        #T_dot = (rT - T) * kP

class Transmission_Pipe(BC.Link):
    def construct_Link(self, kV, **c):
        self.side1 = c['side1']
        self.side2 = c['side2']

        self.calc_funcs = [self.Pipe_diff_f]

        self.Variables = [
            {'name': 'kV', 'unit': 'kg/s/sqrt(kPa)', 'type': 'param', 'val': kV},
            {'name': 'm_dot_max', 'unit': 'kg/s', 'type': 'param', 'val': 1000},
        ]

    def Pipe_diff_f(self):
        T_pipe = ((m_dot > 0) * side1.T + (m_dot <= 0) * side2.T)
        dp = side1.p - side2.p
        m_dot_raw = sign(dp) * sqrt(abs(dp)) * kV
        m_dot =  m_dot_raw * (abs(m_dot_raw) <= m_dot_max) + m_dot_max * (abs(m_dot_raw) > m_dot_max) * sign(m_dot_raw)

        H_dot = m_dot * ((m_dot > 0) * side1.H / side1.m + (m_dot < 0) * side2.H / side2.m)

        side1.m_dot = -m_dot
        side2.m_dot = m_dot
        side1.H_dot = -H_dot
        side2.H_dot = H_dot

class ThermalNode():
    def __init__(self, name, thermal_channels, subnodes):
        self.name = name
        self.sub_nodes={}
        for i, c in enumerate(thermal_channels):
            self.sub_nodes[c.name] = subnodes[i]

class ThermalNode_2C_Producer(BC.Link):

    def construct_Link(self, parent, thermal_channels, subnodes):
        ThermalNode.__init__(self, self.name, thermal_channels, subnodes)

        self.Variables = [
            {'name': 'mflow', 'unit': 'kg/s', 'type': 'state', 'val': 0},
            {'name': 'mflow_set', 'unit': 'kg/s', 'type': 'param', 'val': 0},
            {'name': 'T_produce', 'unit': 'deg.C', 'type': 'state', 'val': 9},
            {'name': 'mflow_max_dot', 'unit': 'kg/s^2', 'type': 'state', 'val': 1/20.0},
            {'name': 'mflow_max', 'unit': 'kg/s', 'type': 'state', 'val': 40},

        ]
        self.parent = parent
        self.Outlet = subnodes[0]
        self.Return = subnodes[1]
        self.calc_funcs = [self.produce]


    def produce(self):
        dT = (Return.T - T_produce + 1e-6)
        dT = (dT < 1) * 1 + (dT > 1) * dT
        mflow_set_raw = parent.P / 4.187 / dT
#        mflow_set_raw = parent.P / 4.187 / (Return.T - T_produce + 1e-6)
        mflow_set = mflow_set_raw * (abs(mflow_set_raw) <= mflow_max) + mflow_max * (abs(mflow_set_raw) > mflow_max)*sign(mflow_set_raw)
        mflow_dot_raw = (mflow_set - mflow) * 0.001
        mflow_dot = mflow_dot_raw * (abs(mflow_dot_raw) <= mflow_max_dot) + mflow_max_dot * (abs(mflow_dot_raw) > mflow_max_dot) * sign(mflow_dot_raw)

        H_dot = mflow * ((mflow > 0) * Return.H / Return.m + (mflow < 0) * Outlet.H / Outlet.m)

        Return.m_dot = -mflow
        Outlet.m_dot = mflow
        Return.H_dot = -H_dot
        Outlet.H_dot = H_dot - parent.P * 1000

class ThermalNode_2C_Consumer(BC.Link):

    def construct_Link(self, parent, kV, thermal_channels, subnodes):
        ThermalNode.__init__(self, self.name, thermal_channels, subnodes)

        self.Variables = [
            {'name': 'kV', 'unit': 'kg/s', 'type': 'state', 'val': 10},


        ]
        self.parent = parent
        self.Outlet = subnodes[0]
        self.Return = subnodes[1]
        self.calc_funcs = [self.consume]

    def consume(self):
        dp = Outlet.p - Return.p
        m_dot = sign(dp) * sqrt(abs(dp)) * kV

        H_dot = m_dot * ((m_dot > 0) * Outlet.H / Outlet.m + (m_dot < 0) * Return.H / Return.m)

        Outlet.m_dot = -m_dot
        Return.m_dot = m_dot
        Outlet.H_dot = -H_dot
        Return.H_dot = H_dot + parent.P * 1000
        # print(kV)

class ThermalLine(BC.Subsystem):
    def construct_Subsystem(self, thermal_channels, thermal_nodes, distances, *Args, **Kwargs):
        self.nodes=[]

        for i, n in enumerate(thermal_nodes):
            self.nodes += [n]
            if i>0:
                for j,c in enumerate(thermal_channels):
                    self.add('Pipe_'+c.name+'_'+prev_n.name+'_'+n.name, Transmission_Pipe, c.kV_m/distances[i],
                         side1=prev_n.sub_nodes[c.name],
                         side2=n.sub_nodes[c.name])

            prev_n = n


class Thermal_Network(BC.Subsystem):
    def construct_Subsystem(self, *args, **kwargs):
        self.producers = []
        self.consumers = []
        self.optim_type = kwargs['optim_type']
        print('Optimization type: ',self.optim_type)
        self.construct_Thermal_Network(*args, **kwargs)
        self.Variables = [
            {'name': 'cost_total', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_kWh', 'unit': 'kr/kWh', 'type': 'state', 'val': 0},
            {'name': 'demand', 'unit': 'kWh/h', 'type': 'state', 'val': 0},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'P_consumed', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'P_produced', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'P_ordered', 'unit': 'kW', 'type': 'param', 'val': 0},
        ]

    def get_producers(self):
        return self.producers

    def set_mix(self, sys_dict, orders):
        #print('orders',orders)
        cost=0
        power=0
        for i, p in enumerate(self.producers):

            cost += orders[i]['cost']
            power += orders[i]['power']
            p.set_production(sys_dict, orders[i])

        #print('Power ordered: ',power)



        cost_kWh = cost/(power+0.001)
        sys_dict[self.gp('P_ordered')] = power
        sys_dict[self.gp('cost_total')] = cost
        sys_dict[self.gp('cost_kWh')] = cost_kWh

    def getDemands(self, sys_dict):
        #print('t: ',sys_dict['t']/3600/24,' days')

        demand = []
        for c in self.consumers:
            demand.append(c.getDemand(sys_dict))

        sys_dict[self.gp('demand')]=sum(demand)

        #print('Power demanded: ', sum(demand))

        return demand

    def getQuotes(self, sys_dict):
        quotes = []
        for p in self.producers:
            quotes.append(p.getQuotes(sys_dict))

        return quotes

    def output_(self, sys_dict):

        cost=0
        acc_cost = 0
        P_prod=0
        for p in self.producers:
            cost += sys_dict[p.gp('cost')]
            acc_cost += sys_dict[p.gp('Acc_cost')]
            P_prod += sys_dict[p.gp('P')]

        P_consumed = 0
        for c in self.consumers:
            #print(c.gp('cost'))
            if c.gp('cost') in sys_dict:
                cost += sys_dict[c.gp('cost')]
                acc_cost += sys_dict[c.gp('Acc_cost')]
            #print(c.name)
            #print(sys_dict[c.gp('P')])
            P_consumed  += sys_dict[c.gp('P')]*c.consumer_sign

        sys_dict[self.gp('P_consumed')] = P_consumed
        sys_dict[self.gp('P_produced')] = P_prod
        sys_dict[self.gp('cost')] = cost
        sys_dict[self.gp('Acc_cost')] = acc_cost

        #Optim
        tyc = time()


        quotes = self.getQuotes(sys_dict)
        demands = self.getDemands(sys_dict)
        demand = max(0, sum(demands))
        # if demand == 0 and not lastdem ==0:
        # lastdem = demand
        # print(quotes)
        # print(demands)
        tic = time()
        #print('quote+demans: ', tic - tyc)

        if self.optim_type == 'Hourly_Mix':
            # opt_mix, orders = generate_orders(guess_mix, demand, quotes, optim=True)
            mix_w_orders = generate_orders(demand, quotes)
            orders = mix_w_orders[0]
            #print('Orders: ', orders)
        elif self.optim_type == 'Fix_Sequence':
            opt_mix, orders = generate_orders_sorted(list(range(len(self.producers))), demand, quotes, optim=False)



        else:
            raise ValueError('Unknown way of setting production: <'+self.optim_type+'>!')
        toc = time()
            #print('optim: ', toc - tic)


            # print('here!: ',orders)
        self.set_mix(sys_dict, orders)

        tac = time()
        #print('setmix: ', tac - toc)

        #return cost, acc_cost

class Hub(BC.Subsystem):
    def construct_Subsystem(self, thermal_channels, **c):
        #print(thermal_channels)
        self.subnodes = [self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0, thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0)]
        #self.side1=self.subnodes[0]
        #self.side2=self.subnodes[1]

        self.thermal_node = ThermalNode(self.name, thermal_channels, self.subnodes)

def calc_cost_quote(quote, power):
    remain_power = power
    i = 0
    cost = 0
    while remain_power > 0:
        power_i = min([quote[i]['Power'], remain_power])
        remain_power -= power_i
        cost += power_i * quote[i]['Cost']
        i += 1
        if i >= len(quote) and remain_power > 0:
            cost += 1e6 * remain_power ** 2
            remain_power = 0

    return cost

class Simple_Consumer(BC.Subsystem):
    def construct_Subsystem(self, CW, P, kV, thermal_channels, **c):
        self.consumer_sign = 1
        self.subnodes = [self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0, thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0)]
        #self.side1=self.subnodes[0]
        #self.side2=self.subnodes[1]

        self.thermal_node = self.add(self.name,ThermalNode_2C_Consumer, self, kV, thermal_channels, self.subnodes)


        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': P},
            {'name': 'P_CS', 'unit': 'kW', 'type': 'param', 'val': P},

        ]

    def getDemand(self,sys_dict):
        sys_dict[self.gp('P')]=sys_dict[self.gp('P_CS')]
        return sys_dict[self.gp('P_CS')] + 10.0*(sys_dict[self.gp('Outlet.T')]-9.0)

def calc_cost_quote(quote, power):
    remain_power = power
    i = 0
    cost = 0
    while remain_power > 0:
        power_i = min([quote[i]['Power'], remain_power])
        remain_power -= power_i
        cost += power_i * quote[i]['Cost']
        i += 1
        if i >= len(quote) and remain_power > 0:
            cost += 1e6 * remain_power ** 2
            remain_power = 0

    return cost

class Fixed_Cost_Producer(BC.Subsystem):
    def construct_Subsystem(self, T, thermal_channels, cost_kWh=1, max_power=500,  **c):

        self.subnodes = [
                         self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0,thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0),
                         ]

        self.Outlet = self.subnodes[0]
        self.Return = self.subnodes[1]

        self.thermal_node = self.add(self.name, ThermalNode_2C_Producer, self, thermal_channels, self.subnodes)

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'state', 'val': 0},
            {'name': 'P_Max', 'unit': 'kW', 'type': 'state', 'val': max_power},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_kWh', 'unit': 'kr/kWh', 'type': 'state', 'val': cost_kWh},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},

        ]

        self.calc_funcs = [self.produce]

    def produce(self):

        #P = P_set

        E_dot=P/3600
        cost = P*cost_kWh/3600
        Acc_cost_dot = cost

    def set_production(self, sys_dict, order):
        quote = order['quote']
        power = order['power']

        sys_dict[self.gp('P')] = min(power*(power>0), sys_dict[self.gp('P_Max')])

    def getQuotes(self, sys_dict):
        quote = {'quote_mode1': [{'Label': 'Dist Cool', 'Power': sys_dict[self.gp('P_Max')], 'Cost': sys_dict[self.gp('cost_kWh')]}],}
        #print(quote)
        return quote

class Disctrict_Cooling(BC.Subsystem):
    def construct_Subsystem(self, T, thermal_channels, cost_kWh=1, max_power=500,  **c):
        self.subscribed_capacity = 1000

        self.subnodes = [
                         self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0,thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0),
                         ]

        self.Outlet = self.subnodes[0]
        self.Return = self.subnodes[1]

        self.thermal_node = self.add(self.name, ThermalNode_2C_Producer, self, thermal_channels, self.subnodes)

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'state', 'val': 0},
            {'name': 'P_Max', 'unit': 'kW', 'type': 'state', 'val': max_power},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_kWh', 'unit': 'kr/kWh', 'type': 'state', 'val': cost_kWh},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},

        ]

        self.calc_funcs = [self.produce]

    def produce(self):

        #P = P_set

        E_dot=P/3600
        cost = P*cost_kWh/3600
        Acc_cost_dot = cost

    def set_production(self, sys_dict, order):
        quote = order['quote']
        power = order['power']
        if power > self.subscribed_capacity:
            self.subscribed_capacity = power
        sys_dict[self.gp('P')] = min(power*(power>0), sys_dict[self.gp('P_Max')])

    def getQuotes(self, sys_dict):

        quote = {'Subscription': [{'Label': 'Dist Cool', 'Power': self.subscribed_capacity, 'Cost': sys_dict[self.gp('cost_kWh')]}, {'Label': 'Dist Cool', 'Power': sys_dict[self.gp('P_Max')], 'Cost': sys_dict[self.gp('cost_kWh')]}],}
        #print(quote)
        return quote

import sim_components.thermodynamics.Heat_Transfer as HT
import numpy as np
class Simple_BHS(BC.Subsystem):
    def construct_Subsystem(self, kP_cost, N_boreholes=63, L_borehole=250, D_borehole=0.044, C_near=6.91e6, C_far=901e6, h_near_far=444, h_far_ground=40, h_brine=30, TG=4.5, T0=5, *args, **c):

        self.N_boreholes = N_boreholes
        self.L_borehole = L_borehole
        self.D_borehole = D_borehole

        self.V_brine = (self.D_borehole/2)**2*np.pi*self.L_borehole*2*self.N_boreholes
        self.C_brine = self.V_brine * 4.187 * 1000
        self.h_brine = h_brine

        self.F_max = N_boreholes*0.8
        self.P_max = N_boreholes*15
        #Add volume near
        #C_near = 6.91e6/60/200*self.N_boreholes*self.L_borehole #kJ/kW
        self.V_near = self.add('V_near', HT.Thermal_Mass, C_near, T0=T0)
        # Add volume far
        #C_far = 901e6/60/200*self.N_boreholes*self.L_borehole #kJ/kW
        self.V_far = self.add('V_far', HT.Thermal_Mass, C_far, T0=T0)
        #Add link conductor near-far
        #h_near_far = 444/60/200*self.N_boreholes*self.L_borehole
        self.L_near_far = self.add('L_near_far', HT.Thermal_Conductor , h_near_far, T0=T0, side1=self.V_near, side2=self.V_far)
        #Add thermal ground
        #h_far_ground = 40/60/200*self.N_boreholes*self.L_borehole
        self.L_far_ground = self.add('L_far_ground', HT.Thermal_Ground, h_far_ground, T_Ground=TG, side1=self.V_far)

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'P_max', 'unit': 'kW', 'type': 'param', 'val': N_boreholes*15},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'T_zero_cost', 'unit': 'oC', 'type': 'param', 'val': 0},
            {'name': 'kP_cost', 'unit': 'kr/oC', 'type': 'param', 'val': kP_cost},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'F_brine', 'unit': 'kg/s', 'type': 'state', 'val': 0},
            {'name': 'F_brine_Max', 'unit': 'kg/s', 'type': 'param', 'val': N_boreholes*0.8},
            {'name': 'T_brine_in', 'unit': 'oC', 'type': 'param', 'val': 0},
            {'name': 'T_brine_out', 'unit': 'oC', 'type': 'param', 'val': 0},
            {'name': 'V_brine', 'unit': 'm3', 'type': 'param', 'val': self.V_brine},
            {'name': 'C_brine', 'unit': 'J/kg/degC', 'type': 'param', 'val': self.C_brine},
            {'name': 'h_C_brine_near', 'unit': '', 'type': 'param', 'val': self.h_brine/self.C_brine},
            {'name': 'HS', 'unit': '', 'type': 'state', 'val': 0},
            {'name': 'CS', 'unit': '', 'type': 'state', 'val': 1},

        ]

        self.calc_funcs = [self.calc]

    def calc(self):

        expo = -V_brine * 1000 / F_brine_Max * h_C_brine_near
        eexpo = exp(expo)
        T_brine_out = V_near.T - (V_near.T - T_brine_in) * eexpo
        P_max = F_brine_Max * 4.187 * (T_brine_in - T_brine_out)

        P = P_max * (abs(P_max) < abs(P_set)) + P_set*(abs(P_max) >= abs(P_set))

        V_near.H_dot = P
        E_dot = P

    def availablePower(self, sys_dict, T_in):

        self.T_near = sys_dict[self.gp('V_near.T')]

        #sys_dict[self.gp('T_brine_in')]=T_in
        T_brine_out = self.T_near - (self.T_near - T_in) * np.exp(-self.V_brine*1000 / self.F_max * self.h_brine/self.C_brine)
        P_avail = min(self.P_max, self.F_max * 4.187 * (T_in - T_brine_out))
        #print(sys_dict[self.gp('kP_cost')])
        cost = (sys_dict[self.gp('V_far.T')] - sys_dict[self.gp('T_zero_cost')])*sys_dict[self.gp('kP_cost')]
        #print(self.name)
        #print('cost: ',cost)
        #print('Tin: ',T_in)
        #print('Power: ', P_avail)
        #if cost > 0:
        #    print('cost?: ',cost)
        return max(0, P_avail), cost

class Simple_Heat_Pump(BC.Subsystem):
    def construct_Subsystem(self, cost_elec_kWh=0.75, cost_backup_kWh=0.6, max_power=100, COP_offset=9, COP_scale=-0.1, **c):

        self.COP_offset = COP_offset
        self.COP_scale = COP_scale


        self.cost_backup_kWh = cost_backup_kWh
#self.T_inlet =


        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_elec_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_elec_kWh},
            {'name': 'cost_backup_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_backup_kWh},

            #{'name': 'cost', 'unit': 'kr/kW/s', 'type': 'state', 'val': 1.0 / 3600.0},
            {'name': 'state', 'unit': '1', 'type': 'state', 'val': 1},
            {'name': 'COP_est', 'unit': '1', 'type': 'param', 'val': 5},
            {'name': 'COP_offset', 'unit': '1', 'type': 'param', 'val': COP_offset},
            {'name': 'COP_scale', 'unit': '1', 'type': 'param', 'val': COP_scale},
            {'name': 'mflow', 'unit': 'kg/s', 'type': 'state', 'val': 5},
            {'name': 'P_Max', 'unit': 'kW', 'type': 'param', 'val': max_power},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'T_HS', 'unit': 'deg C', 'type': 'param', 'val': 40},
            {'name': 'T_in', 'unit': 'deg C', 'type': 'param', 'val': 0},
            {'name': 'P_HS', 'unit': 'deg C', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'h', 'unit': 's', 'type': 'param', 'val': 3600},

            #{'name': 'COP_offset', 'unit': 'kW', 'type': 'param', 'val': COP_offset},
            #{'name': 'COP_slope', 'unit': 'kW', 'type': 'param', 'val': COP_slope},

        ]

        self.calc_funcs = [self.produce]

    def produce(self):
        #P = P_set  # mflow * (Return.T - T_Out) * 4.187

        #P_DH = P_HS - P_set
        dT = T_HS - T_in

        COP_est = COP_offset + COP_scale * dT
        COP_est = COP_est * (COP_est > 1) + (COP_est <= 1)+0.01
        P_set_heat = P_set / (1-1/COP_est)
        P_aux = P_set_heat - P_Max
        P_aux = P_aux * (P_aux > 0)
        P_heat = (P_set_heat - P_aux)
        P = P_heat*(1-1/COP_est)

        E_dot = P / h
        cost_kWh = cost_elec_kWh / COP_est
        cost = P* cost_kWh/h # + P_DH * (P_DH > 0) * cost_backup_kWh)/h
        Acc_cost_dot = cost

    #def set_production(self, sys_dict, order):
        #quote = order['quote']
        #power = order['power']

        #sys_dict[self.gp('P')] = min(power, sys_dict[self.gp('P_Max')])

    def getQuote(self, sys_dict, T_Out_Act, HS_only=False):

        dT=sys_dict[self.gp('T_HS')]-T_Out_Act
        COP_est = max(1,self.COP_offset + self.COP_scale*dT)
        #sys_dict[self.gp('COP_est')]=COP_est
        cost_HP = sys_dict[self.gp('cost_elec_kWh')]/COP_est
        #cost = cost_HP - sys_dict[self.gp('cost_backup_kWh')]
        if sys_dict[self.gp('P_HS')]>0:
            self.quote = [{'Label': 'HP', 'Power': sys_dict[self.gp('P_HS')]*(1-1/COP_est), 'Cost': -self.cost_backup_kWh}]
        else:
            self.quote = []

        if not HS_only:
            if sys_dict[self.gp('P_Max')] - sys_dict[self.gp('P_HS')] >0:
                self.quote.append({'Label': 'HP', 'Power': (sys_dict[self.gp('P_Max')] - sys_dict[self.gp('P_HS')])*(1-1/COP_est), 'Cost': cost_HP})

        return self.quote

class Simple_Heat_Pump_No_BHS(BC.Subsystem):
    def construct_Subsystem(self, thermal_channels, allow_dump=True, cost_elec_kWh=0.75, cost_backup_kWh=0.6, max_power=100, COP_offset=9,
                            COP_scale=-0.1, **c):
        self.consumer_sign = -1
        self.max_power = max_power
        self.allow_dump = allow_dump
        self.cost_backup_kWh = cost_backup_kWh

        self.subnodes = [
        self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0,
                 thermal_channels[0].T0),
        self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0,
                 thermal_channels[1].T0),
        ]

        self.Outlet = self.subnodes[0]
        self.Return = self.subnodes[1]

        self.thermal_node = self.add(self.name, ThermalNode_2C_Producer, self, thermal_channels, self.subnodes)



        self.COP_offset = COP_offset
        self.COP_scale = COP_scale

        # self.T_inlet =

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_elec_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_elec_kWh},
            {'name': 'cost_backup_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_backup_kWh},
            # {'name': 'cost', 'unit': 'kr/kW/s', 'type': 'state', 'val': 1.0 / 3600.0},
            {'name': 'state', 'unit': '1', 'type': 'state', 'val': 1},
            {'name': 'COP_est', 'unit': '1', 'type': 'param', 'val': 5},
            {'name': 'COP_offset', 'unit': '1', 'type': 'param', 'val': COP_offset},
            {'name': 'COP_scale', 'unit': '1', 'type': 'param', 'val': COP_scale},
            {'name': 'mflow', 'unit': 'kg/s', 'type': 'state', 'val': 5},
            {'name': 'P_Max', 'unit': 'kW', 'type': 'param', 'val': max_power},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'T_HS', 'unit': 'deg C', 'type': 'param', 'val': 20},
            {'name': 'T_in', 'unit': 'deg C', 'type': 'param', 'val': 0},

            {'name': 'P_HS', 'unit': 'deg C', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'T_set_HS', 'unit': 'deg C', 'type': 'param', 'val': 40},
            {'name': 'T_set_CS', 'unit': 'deg C', 'type': 'param', 'val': 5},

            # {'name': 'COP_offset', 'unit': 'kW', 'type': 'param', 'val': COP_offset},
            # {'name': 'COP_slope', 'unit': 'kW', 'type': 'param', 'val': COP_slope},

        ]

        self.calc_funcs = [self.produce]

    def produce(self):

        P = P_set  # mflow * (Return.T - T_Out) * 4.187
        E_dot = P / 3600


        dT = T_HS - T_in
        COP_est = COP_offset + COP_scale * dT
        COP_est = COP_est * (COP_est > 1) + (COP_est <= 1)+0.01

        P_Heat = P / (1 - 1 / COP_est)
        P_DH = (P_HS - P_Heat)
        P_DH = P_DH * (P_DH > 0)

        cost_kWh = cost_elec_kWh / COP_est
        cost = (P * cost_kWh + P_DH * cost_backup_kWh) / 3600
        Acc_cost_dot = cost

    def set_production(self, sys_dict, order):
        quote = order['quote']
        power = order['power']

        sys_dict[self.gp('P_set')] = power

        #min(max(power,sys_dict[self.gp('P_HS')]), sys_dict[self.gp('P_Max')])

    def getQuotes(self, sys_dict):

        dT = sys_dict[self.gp('T_HS')] - sys_dict[self.gp('Return.T')]
        COP_est = self.COP_offset + self.COP_scale * dT
        sys_dict[self.gp('COP_est')] = COP_est
        cost = sys_dict[self.gp('cost_elec_kWh')] / COP_est

        if sys_dict[self.gp('P_HS')]>0:
            self.quote = [{'Label': 'HP', 'Power': sys_dict[self.gp('P_HS')]*(1-1/COP_est), 'Cost': -self.cost_backup_kWh}]
        else:
            self.quote = []
        if self.allow_dump:
            if sys_dict[self.gp('P_Max')]>sys_dict[self.gp('P_HS')]:
                self.quote.append(
                    {'Label': 'HP', 'Power': (sys_dict[self.gp('P_Max')] - sys_dict[self.gp('P_HS')])*(1-1/COP_est), 'Cost': cost})

        #print(self.quote)
        return {self.name: self.quote}


    def getDemand(self,sys_dict):
        #print(self.name)
        #print(sys_dict[self.gp('P_HS')])
        P_set = sys_dict[self.gp('P_HS')]
        sys_dict[self.gp('P_set')] =  min(P_set, self.max_power)
        return -P_set #+ 10.0*(sys_dict[self.gp('Outlet.T')]-9.0)

class Simple_Heat_Pump_w_BHS(BC.Subsystem):

    def construct_Subsystem(self, T, thermal_channels, cost_elec_kWh=0.75, cost_backup_kWh=0.6, max_power=100, COP_offset=9, COP_scale=-0.1, kP_cost=10, **c):

        self.subnodes = [
                         self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0,thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0),
                         ]

        self.Outlet = self.subnodes[0]
        self.Return = self.subnodes[1]

        self.thermal_node = self.add(self.name, ThermalNode_2C_Producer, self, thermal_channels, self.subnodes)

        self.add('CS', Transmission_Volume, thermal_channels[0].Fluid, 1, 0,
                 thermal_channels[0].T0),

        self.add('HS', Transmission_Volume, thermal_channels[0].Fluid, .01, 0,
                 thermal_channels[0].T0),

        #Add BTES
        self.BHS = self.add('BHS', Simple_BHS, kP_cost, **c['BHS'])

        self.HP = self.add('HP', Simple_Heat_Pump, cost_elec_kWh, cost_backup_kWh, max_power, COP_offset, COP_scale)

        self.calc_funcs = [self.produce]

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_elec_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_elec_kWh},
            {'name': 'cost_backup_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_backup_kWh},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'P_HS', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'T_HS', 'unit': 'kW', 'type': 'param', 'val': 40},
            {'name': 'T_TN', 'unit': 'kW', 'type': 'param', 'val': T},
            {'name': 'T_set_HS', 'unit': 'deg C', 'type': 'param', 'val': 40},
            {'name': 'T_set_CS', 'unit': 'deg C', 'type': 'param', 'val': 5},
        ]

    def produce(self):
        HP.P_HS = P_HS
        HP.T_HS = T_HS

        P_cool_avail = (Return.T - CS.T) * 1000
        P_cool_avail = (P_cool_avail > 0) * P_cool_avail

        P_dif = P_cool_avail - P_set
        P = P_cool_avail - P_dif * (P_dif > 0)

        err_T_HS = (HS.T - T_set_HS)
        err_T_CS = (CS.T - T_set_CS)

        BHS.P_set = 1000 * (err_T_HS * BHS.HS * (err_T_HS > 0) + err_T_CS * BHS.CS)
        #BHS2.P_set = 1000 * (err_T_HS * BHS2.HS * (err_T_HS > 0) + err_T_CS * BHS2.CS)

        HP.T_in = CS.T
        BHS.T_brine_in = CS.T * BHS.CS + HS.T * BHS.HS
        #BHS2.T_brine_in = CS.T * BHS2.CS + HS.T * BHS2.HS

        P_heat_avail = (HS.T + 5 - T_HS) * 1000
        P_heat_avail = (P_heat_avail > 0) * P_heat_avail

        P_dif_HS = P_HS - P_heat_avail
        P_to_HS = P_heat_avail + P_dif_HS * (P_dif_HS < 0)
        P_DH = P_dif_HS * (P_dif_HS > 0) + HP.P_aux

        CS.H_dot = -BHS.P * BHS.CS - HP.P + P
        HS.H_dot = -BHS.P * BHS.HS*(BHS.P>0) + HP.P_heat - P_to_HS

        cost_kWh = HP.cost_kWh
        cost = HP.cost + P_DH * cost_backup_kWh / 3600
        Acc_cost_dot = cost

    def set_production(self, sys_dict, order):
        #print('backupcost ', sys_dict[self.gp('cost_backup_kWh')])
        quote = order['quote']
        power = order['power']
        COP_est = sys_dict[self.gp('HP.COP_est')]

        P_HP_HS = sys_dict[self.gp('P_HS')]

        sys_dict[self.gp('T_set_CS')] = 5
        #if P_HP_HS>0:
        sys_dict[self.gp('T_set_HS')] = sys_dict[self.gp('T_HS')]
        #else:
        #    sys_dict[self.gp('T_set_HS')] = 10

        #sys_dict[self.gp('HP.P_HS')] = P_HP_HS
        #sys_dict[self.gp('HP.T_HS')] = sys_dict[self.gp('T_HS')]
        sys_dict[self.gp('P_set')] = power

        #print(quote)
        #print()
        #print(quote)
        #print('power: ', power)
        #print('P_HS: ',P_HP_HS)

        if quote == 'A2':

            #sys_dict[self.gp('BHS.P_set')] = power - P_HP_HS*(1-1/COP_est)
            sys_dict[self.gp('BHS.CS')] = 1
            sys_dict[self.gp('BHS.HS')] = 0


            sys_dict[self.gp('HP.P_set')] = P_HP_HS*(1-1/COP_est)
            sys_dict[self.gp('HP.P_HS')] = P_HP_HS

        elif quote == 'B1_2_3':

            #sys_dict[self.gp('BHS.P_set')] = max(0,power-P_HP_HS*(1-1/COP_est))
            sys_dict[self.gp('HP.P_set')] = max(0,max(power, P_HP_HS))*(1-1/COP_est)

            sys_dict[self.gp('HP.P_HS')] = P_HP_HS
            sys_dict[self.gp('BHS.CS')] = 0
            sys_dict[self.gp('BHS.HS')] = 1

        else:
            raise ValueError('Unknown quote: '+quote)

    def getQuotes(self, sys_dict):

        P_BHS_free, cost_BHS_free = self.BHS.availablePower(sys_dict, sys_dict[self.gp('CS.T')])
        #P_BHS2, cost_BHS2 = self.BHS.availablePower(sys_dict, sys_dict[self.gp('Return.T')])

        HP_quote = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')])

        #Quote A2
        self.quotes = {'A2': sorted(HP_quote+[{'Power':P_BHS_free , 'Cost': cost_BHS_free}], key=lambda k: k['Cost'])}

        P_HS = sys_dict[self.gp('HP.P_HS')]
        #if P_HS <= 0:
        P_BHS_active, cost_BHS_active = self.BHS.availablePower(sys_dict, sys_dict[self.gp('HS.T')])

        for i in range(len(HP_quote)):
            if i>0:
                P_this = min(HP_quote[i]['Power'], P_BHS_active)
                P_BHS_active -= P_this
                HP_quote[i]['Power'] = P_this
                HP_quote[i]['Cost'] += cost_BHS_active


        self.quotes.update({'B1_2_3':  sorted(HP_quote, key=lambda k: k['Cost'])})

        return self.quotes


class Simple_Heat_Pump_w_dual_BHS(BC.Subsystem):

    def construct_Subsystem(self, T, thermal_channels, cost_elec_kWh=0.75, cost_backup_kWh=0.6, max_power=100, COP_offset=9, COP_scale=-0.1, kP_cost=10, **c):

        self.subnodes = [
                         self.add('Outlet', Transmission_Volume, thermal_channels[0].Fluid, thermal_channels[0].node_vol, 0,thermal_channels[0].T0),
                         self.add('Return', Transmission_Volume, thermal_channels[1].Fluid, thermal_channels[1].node_vol, 0, thermal_channels[1].T0),
                         ]

        self.Outlet = self.subnodes[0]
        self.Return = self.subnodes[1]

        self.thermal_node = self.add(self.name, ThermalNode_2C_Producer, self, thermal_channels, self.subnodes)

        self.add('CS', Transmission_Volume, thermal_channels[0].Fluid, 1, 0,
                 thermal_channels[0].T0),

        self.add('HS', Transmission_Volume, thermal_channels[0].Fluid, .1, 0,
                 thermal_channels[0].T0),

        #Add BTES
        self.BHS = self.add('BHS', Simple_BHS, kP_cost, **c['BHS1'])
        self.BHS2 = self.add('BHS2', Simple_BHS, kP_cost, **c['BHS2'])
        self.HP = self.add('HP', Simple_Heat_Pump, cost_elec_kWh, cost_backup_kWh, max_power, COP_offset, COP_scale, **c)

        self.calc_funcs = [self.produce]
    #Quotes calc all 17 modes
        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'P_set', 'unit': 'kW', 'type': 'param', 'val': 0},
            {'name': 'Acc_cost', 'unit': 'kr', 'type': 'state', 'val': 0},
            {'name': 'cost_elec_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_elec_kWh},
            {'name': 'cost_backup_kWh', 'unit': 'kr/kWs', 'type': 'param', 'val': cost_backup_kWh},
            {'name': 'cost', 'unit': 'kr/h', 'type': 'param', 'val': 0},
            {'name': 'T_set_HS', 'unit': 'deg C', 'type': 'param', 'val': 40},
            {'name': 'T_set_CS', 'unit': 'deg C', 'type': 'param', 'val': 5},

            #{'name': 'T_Out', 'unit': 'kW', 'type': 'param', 'val': T},

        ]
    #Set_production

    def produce(self):

        HP.P_HS = P_HS
        HP.T_HS = T_HS

        P_cool_avail = (Return.T - CS.T) * 1000
        P_cool_avail = (P_cool_avail > 0) * P_cool_avail

        P_dif = P_cool_avail - P_set
        P = P_cool_avail - P_dif * (P_dif > 0)

        err_T_HS = (HS.T - T_set_HS)
        err_T_CS = (CS.T - T_set_CS)

        BHS.P_set_pre = 1000 * (err_T_HS * BHS.HS * (err_T_HS>0)+ err_T_CS * BHS.CS)
        BHS2.P_set_pre = 1000 * (err_T_HS * BHS2.HS* (err_T_HS>0) + err_T_CS * BHS2.CS)

        #Either both boreholes draw deliver heat or they both receive it
        BHS.P_set = (BHS.P_set_pre <=0)* (BHS2.P_set_pre <=0) *BHS.P_set_pre + (BHS.P_set_pre > 0) * BHS.P_set_pre
        BHS2.P_set = (BHS2.P_set_pre <= 0) * (BHS.P_set_pre <= 0) * BHS2.P_set_pre + (BHS2.P_set_pre > 0) * BHS2.P_set_pre

        HP.T_in = CS.T
        BHS.T_brine_in = CS.T * BHS.CS + HS.T * BHS.HS
        BHS2.T_brine_in = CS.T * BHS2.CS + HS.T * BHS2.HS

        P_heat_avail = (HS.T +2 - T_HS) * 1000
        P_heat_avail = (P_heat_avail > 0) * P_heat_avail

        P_dif_HS = P_HS - P_heat_avail

        P_DH = P_dif_HS * (P_dif_HS > 0) + HP.P_aux
        P_to_HS = P_heat_avail + P_dif_HS * (P_dif_HS < 0)
        CS.H_dot =-BHS.P * BHS.CS - BHS2.P * BHS2.CS - HP.P + P
        HS.H_dot =-BHS.P * BHS.HS - BHS2.P * BHS2.HS + HP.P_heat - P_to_HS



        cost_kWh = HP.cost_kWh
        cost = HP.cost + P_DH * cost_backup_kWh / 3600
        Acc_cost_dot = cost

    def set_production(self, sys_dict, order):
        #print('Power prod ', sys_dict[self.gp('P')])


        quote = order['quote']
        P_TN = order['power']

        COP_est = sys_dict[self.gp('HP.COP_est')]

        sys_dict[self.gp('P_set')] = P_TN
        P_HP_HS = sys_dict[self.gp('P_HS')]

        sys_dict[self.gp('T_set_CS')] = 5
        #if P_HP_HS>0:

        sys_dict[self.gp('T_set_HS')] = sys_dict[self.gp('T_HS')]
        #else:
        #sys_dict[self.gp('T_set_HS')] = 30


        if quote == '2xC':
            #P_BHS_total = P_TN - P_HP_HS
            quotes = self.quotes['2xC']
            sys_dict[self.gp('BHS.CS')] = 1
            sys_dict[self.gp('BHS2.CS')] = 1
            sys_dict[self.gp('BHS.HS')] = 0
            sys_dict[self.gp('BHS2.HS')] = 0
            sys_dict[self.gp('HP.P_set')] = P_HP_HS*(1-1/COP_est)


        elif quote == 'x2H':
            sys_dict[self.gp('BHS.CS')] = 0
            sys_dict[self.gp('BHS2.CS')] = 0
            sys_dict[self.gp('BHS.HS')] = 1
            sys_dict[self.gp('BHS2.HS')] = 1
            sys_dict[self.gp('HP.P_set')] = max(P_TN, P_HP_HS*(1-1/COP_est))

        elif quote == 'x12H':
            quotes = self.quotes['x12H']
            #print(quotes)
            P_HP = P_HP_HS*(1-1/COP_est)
            P_Cool = P_TN - P_HP
            P_HP_HS_rem = P_HP

            sys_dict[self.gp('BHS.CS')] = 1
            sys_dict[self.gp('BHS2.CS')] = 0
            sys_dict[self.gp('BHS.HS')] = 0
            sys_dict[self.gp('BHS2.HS')] = 1

            #Heating need from BHS
            if P_Cool < 0:
            #     #sys_dict[self.gp('BHS.P_set')] = P_Cool
            #     #sys_dict[self.gp('BHS2.P_set')] = P_HP-P_HP_HS
                sys_dict[self.gp('HP.P_set')] = P_HP
            else:
                for q in quotes:
                    if q['Label'] == 'BHS':
                        pass
                    #    P_BHS = min(P_Cool, q['Power'])
                        #sys_dict[self.gp('BHS.P_set')] = P_BHS
                         #print('P_BHS ', P_BHS)
                     #   P_Cool -= P_BHS
            #             #print('P_Cool ', P_Cool)
            #
                    elif q['Label'] == 'HP':
                        P_HP_Extra = min(P_Cool, max(0, q['Power'] - P_HP_HS_rem))
                        P_HP_HS_rem -= max(0, q['Power'])
                        #print('extra x12H',P_HP_Extra)
                        P_HP += P_HP_Extra
                        P_Cool -= P_HP_Extra
            #             #print('P_Cool ',P_Cool)
            #             #sys_dict[self.gp('BHS2.P_set')] = P_HP-P_HP_HS
                        sys_dict[self.gp('HP.P_set')] = P_HP

                    else:
                        raise ValueError('Wrong quote label: ' + q['Label'])
            #
            # if sys_dict[self.gp('BHS.P_set')]>0:
            #     sys_dict[self.gp('BHS.T_brine_in')] = sys_dict[self.gp('Return.T')]
            #     sys_dict[self.gp('HP.T_in')] = sys_dict[self.gp('Return.T')]
            # else:
            #     sys_dict[self.gp('BHS.T_brine_in')] = sys_dict[self.gp('BHS.V_near.T')]-5
            #     sys_dict[self.gp('HP.T_in')] = sys_dict[self.gp('BHS.V_near.T')]-5

            #sys_dict[self.gp('P_set')] = sys_dict[self.gp('BHS.P')] + P_HP
            #if abs(P_Cool)>0.1:
            #    raise ValueError('Some of the BHS energy for x12H not assigned!? '+str(P_Cool)+' kW')


        elif quote == 'x21H':
            quotes = self.quotes['x21H']
            #print(quotes)
            P_HP = P_HP_HS * (1 - 1 / COP_est)
            P_Cool = P_TN - P_HP
            P_HP_HS_rem = P_HP

            sys_dict[self.gp('BHS.CS')] = 0
            sys_dict[self.gp('BHS2.CS')] = 1
            sys_dict[self.gp('BHS.HS')] = 1
            sys_dict[self.gp('BHS2.HS')] = 0


            # Heating need from BHS
            if P_Cool < 0:
                #sys_dict[self.gp('BHS2.P_set')] = P_Cool
                #sys_dict[self.gp('BHS.P_set')] = P_HP-P_HP_HS
                sys_dict[self.gp('HP.P_set')] = P_HP
            else:
                for q in quotes:
                    if q['Label'] == 'HP':
                        P_HP_Extra = min(P_Cool, max(0, q['Power']-P_HP_HS_rem))
                        P_HP_HS_rem -= max(0,q['Power'])
                        #print('extra x21H',P_HP_Extra)
                        P_HP += P_HP_Extra
                        P_Cool -= P_HP_Extra
                        #sys_dict[self.gp('BHS.P_set')] = P_HP-P_HP_HS
                        sys_dict[self.gp('HP.P_set')] = P_HP

                    elif q['Label'] == 'BHS2':
                        pass

                    else:
                        raise ValueError('Wrong quote label: ' + q['Label'])
            #if sys_dict[self.gp('BHS2.P_set')]>0:
            #    sys_dict[self.gp('BHS2.T_brine_in')] = sys_dict[self.gp('Return.T')]
            #    sys_dict[self.gp('HP.T_in')] = sys_dict[self.gp('Return.T')]
            #else:
            #    sys_dict[self.gp('BHS2.T_brine_in')] = sys_dict[self.gp('BHS2.V_near.T')] - 5
            #    sys_dict[self.gp('HP.T_in')] = sys_dict[self.gp('BHS.V_near.T')] - 5
            #if abs(P_Cool)>0.1:
                #ValueError('Some of the BHS energy for x21H not assigned!? '+str(P_Cool)+' kW')

            #sys_dict[self.gp('P_set')] = sys_dict[self.gp('BHS2.P')] + P_HP

        else:
            raise ValueError('Unknown quote: '+quote)


    def getQuotes(self, sys_dict):
        F_Max = 40
        P_BHS_free, cost_BHS_free = self.BHS.availablePower(sys_dict, sys_dict[self.gp('CS.T')])
        P_BHS2_free, cost_BHS2_free = self.BHS2.availablePower(sys_dict, sys_dict[self.gp('CS.T')])

        P_BHS_active, cost_BHS_active = self.BHS.availablePower(sys_dict, sys_dict[self.gp('HS.T')])
        P_BHS2_active, cost_BHS2_active = self.BHS2.availablePower(sys_dict, sys_dict[self.gp('HS.T')])
        P_BHS_active_tot = P_BHS_active + P_BHS2_active

        HP_quote = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')])

        x2c_HP_quote = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')], HS_only=True)

        self.quotes = {'2xC': sorted(x2c_HP_quote+[{'Label':'BHS','Power':P_BHS_free , 'Cost': cost_BHS_free}, {'Label':'BHS2', 'Power':P_BHS2_free , 'Cost': cost_BHS2_free}], key=lambda k: k['Cost'])}

        max_cost_BHS = max(cost_BHS_active, cost_BHS2_active)

        #Both BHS on hot side
        x2H = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')])
        for i in range(len(x2H)):
            if i>0:
                x2H[i]['Cost'] += max_cost_BHS
                P_this = min(HP_quote[i]['Power'], P_BHS_active_tot)
                P_BHS_active_tot -= P_this
                x2H[i]['Power'] = P_this

        self.quotes.update({'x2H':  sorted(x2H, key=lambda k: k['Cost'])})

        #BHS1 on cold, 2 on hot

        x12H = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')])
        #print('x12H ', x12H)
        for i in range(len(HP_quote)):
            if i>0:
                x12H[i]['Cost'] += cost_BHS2_active
                P_this = min(x12H[i]['Power'], P_BHS2_active)
                P_BHS2_active -= P_this
                x2H[i]['Power'] = P_this
        # Both BHS on cold side

        add_list = sorted(x12H + [{'Label':'BHS', 'Power': P_BHS_free, 'Cost': cost_BHS_free}],key=lambda k: k['Cost'])
        #print(add_list)
        self.quotes.update({'x12H': add_list})

        x21H = self.HP.getQuote(sys_dict, sys_dict[self.gp('CS.T')])
        for i in range(len(HP_quote)):
            if i>0:
                x21H[i]['Cost'] += cost_BHS_active
                P_this = min(x21H[i]['Power'], P_BHS_active)
                P_BHS_active -= P_this
                x2H[i]['Power'] = P_this
        # Both BHS on cold side
        self.quotes.update({'x21H': sorted(x21H + [{'Label':'BHS2', 'Power': P_BHS2_free, 'Cost': cost_BHS2_free}],key=lambda k: k['Cost'])})

        #BHS2 on cold, 1 on hot

        #print(self.quotes)
        return self.quotes

